// backend/src/services/agentService.ts

import { Prisma } from "@prisma/client"; // Import Prisma types
import bcrypt from "bcryptjs";
import { Agent, AgentCredentials } from "../../interfaces/Agent";
import { AgentRepository } from "repositories/agent/agentRepository";
// import { AgentRepository } from "../../repositories/agent/agentRepository";
// import { AgentRepository } from "../../repositories/user/userAgentRepository";


export class agentService {
  private agentRepository: AgentRepository;

  constructor(agentRepository: AgentRepository) {
    this.agentRepository = agentRepository;
  }

  public async createAgent(
    agentData: Agent,
    credentialsData: AgentCredentials = {} as AgentCredentials
  ): Promise<Agent> {
    try {
      console.log("Service - Creating Agent:", agentData);

      if (!agentData.email) {
        throw new Error("Agent email is required.");
      }

      // Validate password if provided
      if (credentialsData.password && credentialsData.password.length < 6) {
        throw new Error("Password must be at least 6 characters long.");
      }

      // Hash password if it exists
      if (credentialsData.password) {
        credentialsData.password = await bcrypt.hash(
          credentialsData.password,
          10
        );
      } else if (agentData.agentPrivilege) {
        console.warn(
          `Agent ${agentData.email} created with privileges but no password.`
        );
      }

      // Call repository to save the agent
      const newAgent = await this.agentRepository.createAgent(
        agentData,
        credentialsData
      );
      return newAgent;
    } catch (error) {
      console.error("Error creating agent in service:", error);
      throw error;
    }
  }

  public async updateAgent(
    agentId: string,
    agentData: Partial<Agent>,
    credentialsData?: Partial<AgentCredentials>
  ): Promise<Agent> {
    try {
      // Hash password if provided in the update
      if (credentialsData?.password) {
        if (credentialsData.password.length < 6) {
          throw new Error("Password must be at least 6 characters long.");
        }
        const hashedPassword = await bcrypt.hash(credentialsData.password, 10); // Hash with salt rounds
        credentialsData.password = hashedPassword; // Replace plain password with hash
      }

      const updatedAgent = await this.agentRepository.updateAgent(
        agentId,
        agentData,
        credentialsData
      );
      if (!updatedAgent) {
        throw new Error(`Agent with ID ${agentId} not found for update.`);
      }
      return updatedAgent;
    } catch (error) {
      console.error("Error updating agent in service:", error);
      // Handle potential unique constraint errors during update
      if (
        error instanceof Prisma.PrismaClientKnownRequestError &&
        error.code === "P2002"
      ) {
        const fields = (error.meta as any)?.target;
        throw new Error(
          `Agent update failed: The following field(s) must be unique: ${fields?.join(
            ", "
          )}`
        );
      }
      // Check if it's a Prisma specific error for not found (already handled by repo returning null/throwing)
      if (
        error instanceof Prisma.PrismaClientKnownRequestError &&
        error.code === "P2025"
      ) {
        throw new Error(`Agent update failed: Record to update not found.`);
      }
      throw error; // Re-throw other errors
    }
  }

  public async getAllAgents(): Promise<Agent[]> {
    try {
      const agents = await this.agentRepository.getAllAgents();
      return agents;
    } catch (error) {
      console.error("Error getting agents in service:", error);
      throw error;
    }
  }
  public async softDeleteAgent(
    agentId: string): Promise<Agent | null> {
      console.log("Service - Soft Deleting Agent:", agentId);

    try {
      const deletedAgent = await this.agentRepository.softDeleteAgent(agentId);
      if (!deletedAgent) {
        // Repository might return null if not found, handle it here
        console.warn(`Agent with ID ${agentId} not found for deletion.`);
        return null;
        // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
      }
      return deletedAgent;
    }
    catch (error) {
      console.error("Error soft deleting agent in service:", error);
      // Handle potential unique constraint errors during deletion
      return null; // Ensure a return value in case of an error
    }
  }
  public async restoreAgent(agentId: string): Promise<Agent | null> {
    try {
      const restoredAgent = await this.agentRepository.restoreAgent(agentId);
      if (!restoredAgent) {
        console.warn(`Agent with ID ${agentId} not found for restore.`);
        return null;
      }
      return restoredAgent;
    } catch (error) {
      console.error("Error restoring agent in service:", error);
      throw error;
    }
  }

  public async deleteAgent(agentId: string): Promise<Agent | null> {
    try {
      const deletedAgent = await this.agentRepository.deleteAgent(agentId);
      if (!deletedAgent) {
        // Repository might return null if not found, handle it here
        console.warn(`Agent with ID ${agentId} not found for deletion.`);
        return null;
        // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
      }
      return deletedAgent;
    } catch (error) {
      console.error("Error deleting agent in service:", error);
      if (
        error instanceof Prisma.PrismaClientKnownRequestError &&
        error.code === "P2025"
      ) {
        console.warn(
       `   Agent with ID ${agentId} not found for deletion (Prisma Error).`
        );
        return null;
        // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
      }
      throw error;
    }
  }
}