//userBookingService
import { BookingRepository } from "../../repositories/user/UserbookingRepository";
import prisma from "../../config/prismaClient";
import { LeadStatus, LeadStage, BookingPriceDetailsAccommodation } from '@prisma/client'; // Import LeadStatus enum

export class BookingService {
    private bookingRepository: BookingRepository;

    constructor(bookingRepository?: BookingRepository) {
        this.bookingRepository = bookingRepository || new BookingRepository();
    }

    async fetchQuotationWithDetails(quotationId: string) {
        try {
            return await this.bookingRepository.fetchQuotationWithDetails(quotationId);
        } catch (error) {
            console.error("Error fetching quotation details:", error);
            throw new Error("Failed to fetch quotation details");
        }
    }

    async createBookingFromQuotation(quotationId: string) {
        try {
            const quotation = await this.bookingRepository.fetchQuotationWithDetails(quotationId);
            if (!quotation) {
                throw new Error("Quotation not found");
            }

            return await this.bookingRepository.handleCreateBooking(quotation);
        } catch (error) {
            console.error("Error creating booking from quotation:", error);
            throw new Error("Failed to create booking from quotation");
        }
    }

    async getAllBookings(userId: string) {
        try {
            return await this.bookingRepository.getAllBookings(userId);
        } catch (error) {
            console.error("Error fetching all bookings:", error);
            throw new Error("Failed to retrieve bookings");
        }
    }

    async getBookingById(id: string) {
        try {
            return await this.bookingRepository.getBookingById(id);
        } catch (error) {
            console.error("Error fetching booking by ID:", error);
            throw new Error("Failed to retrieve booking");
        }
    }

    async updateBookingStatus(bookingId: string, status: LeadStatus, userName: string) {
        try {
          if (!Object.values(LeadStatus).includes(status)) {
            throw new Error('Invalid status value');
          }
            const updatedBooking = await prisma.booking.update({
                where: { id: bookingId },
                data: {
                    lead: {
                        update: {
                            status: status,
                        },
                    },
                },
                include: {
                    lead: true,
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                },
            });

            return updatedBooking;
        } catch (error) {
            console.error('Error updating booking status:', error);
            throw new Error('Failed to update booking status');
        }
    }

    async updateBookingStage(bookingId: string, stage: LeadStage, notes: string, userName: string) {
    try {
            if (!Object.values(LeadStage).includes(stage)) {
                throw new Error('Invalid stage value');
            }

            const updatedBooking = await prisma.booking.update({
                where: { id: bookingId },
                data: {
                    lead: {
                        update: {
                            stage: stage,
                        },
                    },
                },
                include: {
                    lead: true,
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                },
            });

            return updatedBooking;
        } catch (error) {
            console.error('Error updating booking stage:', error);
            throw new Error('Failed to update booking stage');
        }
    }

    async updateAccommodationInBooking(
        bookingId: string,
        accommodationId: string,
        updateData: {
            hotel?: string;
            room?: string;
            supplierName?: string;
            supplierContact?: string;
            travelDate?: string;
            nights?: number;
            quotePrice?: number | null;
            actualPrice?: number | null;
            confirmationNo?: string;
            details?: string;
            quantity?: number;
            priceDetailsAccommodation?: BookingPriceDetailsAccommodation[];
        }
    ) {
        try {
            const booking = await this.bookingRepository.findBookingWithAccommodations(bookingId);
            if (!booking) throw new Error("Booking not found");
    
            const accommodation = booking.accommodations.find(
                (acc) => acc.id === accommodationId
            );
            if (!accommodation) throw new Error("Accommodation not found in this booking");
    
            const updated = await this.bookingRepository.updateAccommodation(accommodationId, updateData);
    
            return updated;
        } catch (error) {
            console.error("Service Error:", error);
            throw new Error("Failed to update accommodation in booking");
        }
    }
    
    async updateItinerary(
        bookingId: string,
        itineraryId: string,
        updateData: {
            date?: string;
            label?: string;
            category?: string;
            quotedPrice?: number | null;
            actualPrice?: number | null;
            supplierId?: string;
            supplierName?: string;
            supplierContact?: string;
            confirmationNo?: string;
            details?: string;
            quantity?: number;
            priceDetailsList?: any[];
        }
    ) {
        // Verify booking exists
        const booking = await this.bookingRepository.getBookingById(bookingId);
        if (!booking) {
            throw new Error("Booking not found");
        }
    
        // Verify itinerary belongs to this booking
        const itinerary = booking.itineraries.find(it => it.id === itineraryId);
        console.log("Itinerary found:", itinerary);
        if (!itinerary) {
            throw new Error("Itinerary not found in this booking");
        }
    
        return this.bookingRepository.updateItinerary(itineraryId, updateData);
    }
    
    }