// src/services/notification.service.ts
import prisma from "../../config/prismaClient";
import { getIO } from "../../socket/socket";

/** Payload for a new notification */
interface CreateNotificationDto {
  type: string;
  title: string;
  message: string;
  entityType: string;
  entityId: string;
  /** "admin" for super-admin broadcast OR the userId of the recipient */
  recipientId: string;
}

export class NotificationService {
  private userRoom = (userId: string) => `user_${userId}`;

  /** Send user notification */
  async notifyUser(userId: string, dto: Omit<CreateNotificationDto, "recipientId">) {
    return this.saveAndEmit({ ...dto, recipientId: userId }, this.userRoom(userId));
  }

  async notifyAdmin(dto: Omit<CreateNotificationDto, "recipientId">) {
    return this.saveAndEmit({ ...dto, recipientId: "admin" }, "admin");
  }

  private async saveAndEmit(data: CreateNotificationDto, room: string) {
    const io = getIO(); // ✅ fetch it inside method
    const notification = await prisma.notification.create({ data });

    io.to(room).emit("receive_notification", {
      ...data,
      createdAt: notification.createdAt,
    });

    console.log(`📨 emitted to [${room}]`, {
      type: data.type,
      title: data.title,
    });

    return notification;
  }

  getAdminNotifications() {
    return prisma.notification.findMany({
      where: { recipientId: "admin" },
      orderBy: { createdAt: "desc" },
    });
  }

  getUserNotifications(userId: string) {
    return prisma.notification.findMany({
      where: { recipientId: userId },
      orderBy: { createdAt: "desc" },
    });
  }

  markAsRead(id: string) {
    return prisma.notification.update({ where: { id }, data: { isRead: true } });
  }

  markAllAsReadForAdmin() {
    return prisma.notification.updateMany({
      where: { recipientId: "admin", isRead: false },
      data: { isRead: true } });
  }
}

