import { Prisma } from "@prisma/client";
import { error } from "console";
import { Agent, AgentCredentials } from "../../interfaces/Agent";
import { AgentRepository } from "../../repositories/agent/agentRepository";
import { HttpError } from "../../utils/HttpError";
import bcrypt from 'bcrypt'; // Import bcrypt

export class agentService {
    private agentRepository: AgentRepository;

    constructor(agentRepository: AgentRepository) {
        this.agentRepository = agentRepository;
    }

    public async createAgent(
        agentData: Agent,
        credentialsData?: AgentCredentials
    ): Promise<Agent> {
        try {
            // 🔍 Check email conflict
            if (agentData.email) {
                const existingEmail = await this.agentRepository.getAgentByEmail(agentData.email);
                if (existingEmail) {
                    throw new HttpError("An agent with this email already exists.", 409);
                }
            }

            // 🔍 Check phone conflict
            if (agentData.phone) {
                const existingPhone = await this.agentRepository.getAgentByPhone(agentData.phone);
                if (existingPhone) {
                    throw new HttpError("An agent with this phone number already exists.", 409);
                }
            }

            // 🔍 Check agentCode conflict
            if (agentData.agentCode) {
                const existingCode = await this.agentRepository.getAgentByAgentCode(agentData.agentCode);
                if (existingCode) {
                    throw new HttpError("An agent with this agent code already exists.", 409);
                }
            }

            // Hash the password if credentials are provided
            if (credentialsData?.password) {
                const hashedPassword = await bcrypt.hash(credentialsData.password, 10); // 10 is the salt rounds
                credentialsData.password = hashedPassword; // Store the HASHED password
            }

            return await this.agentRepository.createAgent(agentData, credentialsData);
        } catch (error: any) {
            if (error instanceof HttpError) {
                throw error; // Re-throw HttpError
            } else {
                console.error("AgentService Create Error:", error);
                throw new HttpError("Failed to create agent", 500);
            }
        }
    }

    public async updateAgent(
        agentId: string,
        agentData: Partial<Agent>,
        credentialsData?: Partial<AgentCredentials>
    ): Promise<Agent | null> {
        try {
            // Conditionally hash the password only if a new password is provided
            if (credentialsData?.password) {
                const hashedPassword = await bcrypt.hash(credentialsData.password, 10);
                credentialsData.password = hashedPassword; // Store the HASHED password
            } else {
                delete credentialsData?.password;
            }

            return await this.agentRepository.updateAgent(
                agentId,
                agentData,
                credentialsData
            );
        } catch (error: any) {
            if (error instanceof HttpError) {
                throw error; // Re-throw HttpError
            } else {
                console.error("AgentService Update Error:", error);
                throw new HttpError("Failed to update agent", 500);
            }
        }
    }

    public async getAllAgents(): Promise<Agent[]> {
        try {
            const agents = await this.agentRepository.getAllAgents();
            return agents;
        } catch (error) {
            console.error("Error getting agents in service:", error);
            throw error;
        }
    }

    public async softDeleteAgent(agentId: string): Promise<Agent | null> {
        console.log("Service - Soft Deleting Agent:", agentId);

        try {
            const deletedAgent = await this.agentRepository.softDeleteAgent(agentId);
            if (!deletedAgent) {
                // Repository might return null if not found, handle it here
                console.warn(`Agent with ID ${agentId} not found for deletion.`);
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            return deletedAgent;
        } catch (error) {
            console.error("Error soft deleting agent in service:", error);
            // Handle potential unique constraint errors during deletion
            return null; // Ensure a return value in case of an error
        }
    }

    public async restoreAgent(agentId: string): Promise<Agent | null> {
        try {
            const restoredAgent = await this.agentRepository.restoreAgent(agentId);
            if (!restoredAgent) {
                console.warn(`Agent with ID ${agentId} not found for restore.`);
                return null;
            }
            return restoredAgent;
        } catch (error) {
            console.error("Error restoring agent in service:", error);
            throw error;
        }
    }

    public async deleteAgent(agentId: string): Promise<Agent | null> {
        try {
            const deletedAgent = await this.agentRepository.deleteAgent(agentId);
            if (!deletedAgent) {
                // Repository might return null if not found, handle it here
                console.warn(`Agent with ID ${agentId} not found for deletion.`);
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            return deletedAgent;
        } catch (error) {
            console.error("Error deleting agent in service:", error);
            if (
                error instanceof Prisma.PrismaClientKnownRequestError &&
                error.code === "P2025"
            ) {
                console.warn(
                    `   Agent with ID ${agentId} not found for deletion (Prisma Error).`
                );
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            throw error;
        }
    }
}