// backend/src/services/admin/bookingService.ts
import { BookingRepository } from "../../repositories/admin/bookingRepository";
import prisma from "../../config/prismaClient";
import {
  LeadStatus,
  LeadStage,
  BookingPriceDetailsAccommodation, // This is the Prisma model type
  BookingPriceDetailItinerary,    // This is the Prisma model type
  FinalItineraryItem,
  Prisma, // For input types
} from '@prisma/client';

// Define the shape of updateData for updateAccommodationInBooking for clarity
// This should match what the controller sends and what the repository expects for its own DTO
type AccommodationUpdateServiceData = {
  hotel?: string;
  room?: string;
  supplierName?: string;
  supplierContact?: string;
  travelDate?: string;
  nights?: number;
  quotePrice?: number | null;
  actualPrice?: number | null;
  confirmationNo?: string; // Consistent naming from controller
  details?: string;
  quantity?: number;
  priceDetailsAccommodation?: Prisma.BookingPriceDetailsAccommodationCreateManyAccommodationInput[]; // Use Prisma input type
};

type ItineraryUpdateServiceData = {
    date?: string;
    label?: string;
    category?: string;
    quotePrice?: number | null;
    actualPrice?: number | null;
    supplierId?: string;
    supplierName?: string;
    supplierContact?: string;
    confirmationNo?: string; // Consistent naming from controller
    details?: string;
    priceDetailsList?: Prisma.BookingPriceDetailItineraryCreateManyItineraryInput[]; // Use Prisma input type
};


export class BookingService {
  private bookingRepository: BookingRepository;

  constructor(bookingRepository?: BookingRepository) {
    this.bookingRepository = bookingRepository || new BookingRepository();
  }

  async fetchQuotationWithDetails(quotationId: string) {
    try {
      return await this.bookingRepository.fetchQuotationWithDetails(quotationId);
    } catch (error) {
      console.error("Error fetching quotation details:", error);
      throw new Error("Failed to fetch quotation details");
    }
  }

  async createBookingFromQuotation(quotationId: string) {
    try {
      const quotation = await this.bookingRepository.fetchQuotationWithDetails(quotationId);
      if (!quotation) {
        throw new Error("Quotation not found");
      }
      return await this.bookingRepository.handleCreateBooking(quotation);
    } catch (error) {
      console.error("Error creating booking from quotation:", error);
      throw new Error("Failed to create booking from quotation");
    }
  }

  async getAllBookings() {
    try {
      return await this.bookingRepository.getAllBookings();
    } catch (error) {
      console.error("Error fetching all bookings:", error);
      throw new Error("Failed to retrieve bookings");
    }
  }

  async getBookingById(id: string) {
    try {
      const booking = await this.bookingRepository.getBookingById(id);
      if (!booking) {
        throw new Error("Booking not found");
      }
      return booking;
    } catch (error: any) { // Catch specific error type if possible
      console.error("Error fetching booking by ID in service:", error.message);
      if (error.message === "Booking not found") {
        throw error;
      }
      throw new Error("Failed to retrieve booking");
    }
  }

  async updateBookingStatus(bookingId: string, status: LeadStatus, userName: string) {
    try {
      if (!Object.values(LeadStatus).includes(status)) {
        throw new Error('Invalid status value');
      }
      // Ensure your prisma.booking.update includes the necessary relations if you return them
      return await prisma.booking.update({
        where: { id: bookingId },
        data: {
          lead: {
            update: {
              status: status,
            },
          },
        },
        include: { // Add includes if the FE expects the full booking object back
          lead: true,
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          hotelVouchers: true,
        },
      });
    } catch (error) {
      console.error('Error updating booking status:', error);
      throw new Error('Failed to update booking status');
    }
  }

  async updateBookingStage(bookingId: string, stage: LeadStage, notes: string, userName: string) {
    try {
      if (!Object.values(LeadStage).includes(stage)) {
        throw new Error('Invalid stage value');
      }
      return await prisma.booking.update({
        where: { id: bookingId },
        data: {
          lead: {
            update: {
              stage: stage,
              // If you have a FollowUp model for stage changes, create it here
            },
          },
        },
        include: { // Add includes if the FE expects the full booking object back
          lead: true,
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          hotelVouchers: true,
        },
      });
    } catch (error) {
      console.error('Error updating booking stage:', error);
      throw new Error('Failed to update booking stage');
    }
  }

  // This is the critical method for the TS2554 error
  async updateAccommodationInBooking(
    bookingId: string,
    accommodationId: string,
    updateData: AccommodationUpdateServiceData // Using the defined type
  ) {
    try {
      const booking = await this.bookingRepository.findBookingWithAccommodations(bookingId);
      if (!booking) throw new Error("Booking not found");

      const accommodationExists = booking.accommodations.find(
        (acc) => acc.id === accommodationId
      );
      if (!accommodationExists) throw new Error("Accommodation not found in this booking");

      // Pass data to repository, ensuring field names match (e.g., confirmationNo)
      // The repository's updateAccommodation expects an object that matches its BookingAccommodationUpdateData type
      return await this.bookingRepository.updateAccommodation(accommodationId, {
        hotel: updateData.hotel,
        room: updateData.room,
        supplierName: updateData.supplierName,
        supplierContact: updateData.supplierContact,
        travelDate: updateData.travelDate,
        nights: updateData.nights,
        quotePrice: updateData.quotePrice,
        actualPrice: updateData.actualPrice,
        confirmationNo: updateData.confirmationNo, // Map from service DTO to repo DTO if names differ
        details: updateData.details,
        quantity: updateData.quantity,
        priceDetailsAccommodation: updateData.priceDetailsAccommodation,
      });
    } catch (error: any) {
      console.error("Service Error in updateAccommodationInBooking:", error.message);
      throw new Error("Failed to update accommodation in booking");
    }
  }

  async updateItinerary(
    bookingId: string,
    itineraryId: string,
    updateData: ItineraryUpdateServiceData
  ) {
    const booking = await this.bookingRepository.getBookingById(bookingId);
    if (!booking) {
      throw new Error("Booking not found");
    }

    const itineraryExists = booking.itineraries.find(it => it.id === itineraryId);
    if (!itineraryExists) {
      throw new Error("Itinerary not found in this booking");
    }

    return this.bookingRepository.updateItinerary(itineraryId, {
        date: updateData.date,
        label: updateData.label,
        category: updateData.category,
        supplierId: updateData.supplierId,
        supplierName: updateData.supplierName,
        supplierContact: updateData.supplierContact,
        quotePrice: updateData.quotePrice,
        actualPrice: updateData.actualPrice,
        confirmationNo: updateData.confirmationNo, // Repo expects confirmationNumber based on its DTO
        details: updateData.details,
        priceDetailsList: updateData.priceDetailsList,
    });
  }

  async getFinalItineraryItems(bookingId: string) {
    return this.bookingRepository.getFinalItineraryItemsByBookingId(bookingId);
  }

  async createFinalItineraryItem(
    bookingId: string,
    data: Omit<FinalItineraryItem, 'id' | 'bookingId' | 'createdAt' | 'updatedAt' | 'booking'>
  ) {
    return this.bookingRepository.createFinalItineraryItem(bookingId, data);
  }

  async updateFinalItineraryItem(
    itemId: string,
    data: Partial<Omit<FinalItineraryItem, 'id' | 'bookingId' | 'createdAt' | 'updatedAt' | 'booking'>>
  ) {
    return this.bookingRepository.updateFinalItineraryItem(itemId, data);
  }

  async deleteFinalItineraryItem(itemId: string) {
    return this.bookingRepository.deleteFinalItineraryItem(itemId);
  }

  async generateInitialFinalItinerary(bookingId: string) {
    return this.bookingRepository.generateInitialFinalItineraryForBooking(bookingId);
  }
}