// backend/src/server.ts
import cors from "cors";
import dotenv from "dotenv";
import express, { NextFunction, Request, Response } from "express";
import path from "path";
import adminRoutes from "./routes/adminRoutes";
import agentRoutes from "./routes/agentRoutes";
import userRoutes from "./routes/userRoutes";
import http from "http";
import { Server } from "socket.io";
import {  initSocket } from "./socket/socket";
import { NotificationService } from "./services/common/notificationService";

dotenv.config();

const app = express();
const server = http.createServer(app);

const allowedOrigins = [
  "https://ssworldcrm.sixthstar.in",
  "https://ktholidayscrm.sixthstar.in",
  "http://localhost:5173" // Keep localhost for development!
];

const io = new Server(server, {
  cors: {
    origin: allowedOrigins, // Use the array for Socket.IO as well
    credentials: true,
  },
});

initSocket(io);
// const notificationService = new NotificationService();

app.use(
  cors({
    origin: (origin, callback) => {
      if (!origin || allowedOrigins.includes(origin)) {
        callback(null, true);
      } else {
        callback(new Error("Not allowed by CORS"));
      }
    },
    credentials: true,
    allowedHeaders: [
      "Authorization",
      "Content-Type",
      "x-user-role",
      "x-refresh-token",
    ],
    exposedHeaders: ["x-access-token", "x-refresh-token"],
    methods: ["GET", "POST", "PUT", "DELETE", "OPTIONS", "PATCH"],
  })
);

app.use(express.json());

const agentImagesPath = path.join(__dirname, "assets", "agent-images");
app.use("/agent-images", express.static(agentImagesPath));

const packagesPath = path.join(__dirname, "assets", "packages");
app.use("/assets/packages", express.static(packagesPath));

// Load Routes Dynamically
async function loadRoutes() {

  app.use("/admin", adminRoutes);
  app.use("/agent", agentRoutes);
  app.use("/user", userRoutes);
}

// Call loadRoutes to load the routes
loadRoutes().catch(err => console.error("Failed to load routes", err));

app.use((err: any, req: Request, res: Response, next: NextFunction) => {
  console.error(err.stack);
  res.status(500).json({ message: err.message || "Internal Server Error" });
});

app.use((req: Request, res: Response) => {
  res.status(404).json({ message: "Route not found" });
});

const PORT = process.env.PORT || 5000;

server.listen(PORT, () => console.log(
   ` Server running on port ${PORT}. Accessible via ${
      process.env.SERVER_URL || "http://localhost"
    }`
) );