import express, { NextFunction, Request, Response } from "express";
// import { authenticate } from "../middleware/authMiddleware";
import { agentUpload, documentUpload } from "../config/multerConfig";
import { AgentController } from "../controllers/user/userAgentController"; // Check the import path correct
import { UserLeadController } from "../controllers/user/UserLeadController";
import { userLogin } from "../controllers/user/userLoginController";
import { QuotationController } from "../controllers/user/userquotationController";
import { authenticateUser } from "../middlewares/authMiddleware1";
import { authorizeRole } from "../middlewares/roleMiddleware";
import { AgentRepository } from "../repositories/agent/agentRepository"; //This line is the problem
import { UserLeadRepository } from "../repositories/user/userLeadRepository";
import { QuotationRepository } from "../repositories/user/userquotationRepository";
import { agentService } from "../services/agent/agentService";
import { UserLeadService } from "../services/user/UserLeadService";
import { QuotationService } from "../services/user/userquotationService";
import { DocumentRepository } from "../repositories/user/documentRepository";
import { DocumentService } from "../services/user/documentService";
import { DocumentController } from "../controllers/user/documentController";
// import { BookingRepository } from "../repositories/user/UserBookingRepository";
import { BookingService } from "../services/user/UserBookingService";
import { BookingController } from "../controllers/user/UserBookingController";
import { BookingRepository } from "../repositories/user/UserbookingRepository";
import { getOperationDashboardData, getTop5Bookings, getBookingLeadOverview } from "../controllers/user/operationDashboardController";
import { GuideController } from "../controllers/user/userGuideController";
import { TourController } from "../controllers/user/userTourController";
import prisma from "../config/prismaClient";
import { UserAssessmentRepository } from "../repositories/user/userAssessmentRepository";
import { UserAssessmentService } from "../services/user/userAssessmentService";
import { UserAssessmentController } from "../controllers/user/userAssessmentContoller";
import { UserDashboardController } from "../controllers/user/UserDashboardController";
import { UserDashboardService } from "../services/user/UserTeamDashboardService";
import { UserTeamDashboardRepository } from "../repositories/user/UserTeamDashboardRepository";
import { getAssessmentDashboardData, getAssessmentLeadOverview } from "../controllers/user/assessmentDashboardController";
import { getUserNotifications } from "../controllers/common/notificationController";
// import { getUserNotifications } from "../controllers/common/notificationController";

const userDashboardRepository = new UserTeamDashboardRepository();
const userDashboardService = new UserDashboardService();
const userDashboardController = new UserDashboardController(userDashboardService);

const userLeadRepository = new UserLeadRepository();
const userLeadService = new UserLeadService(userLeadRepository);
const userLeadController = new UserLeadController(userLeadService);


const userAssessmentRepository = new UserAssessmentRepository();
const userAssessmentService = new UserAssessmentService(userAssessmentRepository);  
const userAssessmentController = new UserAssessmentController(userAssessmentService); // Create an instance of UserAssessmentController

const quotationRepository = new QuotationRepository();
const quotationService = new QuotationService(quotationRepository);
const quotationController = new QuotationController(quotationService);

const agentRepository = new AgentRepository();
const agentservice = new agentService(agentRepository);
const agentController = new AgentController(agentservice);

const documentRepository = new DocumentRepository();
const documentService = new DocumentService(documentRepository);
const documentController = new DocumentController(documentService);

const bookingRepository = new BookingRepository();
const bookingService = new BookingService(bookingRepository);
const bookingController = new BookingController(bookingService);



const guideController = new GuideController(); // Create an instance of GuideController

const tourController = new TourController();

// const operationDashboardRouter = express.Router();


const userRouter = express.Router();

userRouter.post("/login", userLogin); // added middleware
userRouter.get(
  "/userdashboard",
  authenticateUser,
  authorizeRole(["user"]),
  (req: Request, res: Response, next: NextFunction) => {
    try {
      userDashboardController.getUserDashboardData(req, res, next);
    } catch (error) {
      console.error("Error in userRouter:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);

userRouter.get(
  "/assessmentdashboard",
  authenticateUser, 
  authorizeRole(["user"]),
  (req: Request, res: Response, next: NextFunction) => {
    try {
      getAssessmentDashboardData(req, res);
    } catch (error) {
      console.error("Error in userRouter:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);

userRouter.get(
  "/assessmentleadoverview/:timeframe?",
  authenticateUser,
  authorizeRole([ "user"]),
  (req: Request, res: Response, next: NextFunction) => {
    try {
       getAssessmentLeadOverview(req, res);
    } catch (error) {
      console.error("Error in userRouter:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);

userRouter.post(
  "/leads",
  authenticateUser,
  authorizeRole(["user"]),
  (req, res, next) => {
    try {
      userLeadController.createLead(req, res, next);
    } catch (error) {
      console.error("Error in userRouter:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);
userRouter.get(
  "/leads",
  authenticateUser,
  authorizeRole(["user"]),
  (req, res, next) => {
    try {
      userLeadController.getUserLeads(req, res, next);
    } catch (error) {
      console.error("Error in userRouter:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);
userRouter.get(
  "/leads/:leadId",
  authenticateUser,
  authorizeRole(["user"]),
  (req, res, next) => {
    try {
      userLeadController.getLeadById(req, res, next);
    } catch (error) {
      console.error("Error in userRouter:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);
userRouter.put(
  "/leads/:leadId",
  authenticateUser,
  authorizeRole(["user"]),
  (req, res, next) => {
    try {
      userLeadController.updateLead(req, res, next);
    } catch (error) {
      console.error("Error in userRouter:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);
userRouter.delete(
  "/leads/:leadId",
  authenticateUser,
  authorizeRole(["user"]),
  (req, res, next) => {
    try {
      userLeadController.deleteLead(req, res, next);
    } catch (error) {
      console.error("Error in userRouter:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);

// Quotation routes for users
userRouter.post(
  "/createquotations",
  authenticateUser,
  authorizeRole(["user"]),
  (req, res, next) => {
    quotationController.createQuotation(req, res).catch(next);
  }
);
userRouter.get(
  "/quotations",
  authenticateUser,
  authorizeRole(["user"]),
  async (req, res, next) => {
    try {
      await quotationController.getQuotations(req, res);
    } catch (error) {
      next(error);
    }
  }
);
userRouter.get(
  "/quotations/:id",
  authenticateUser,
  authorizeRole(["user"]),
  async (req, res, next) => {
    try {
      await quotationController.getQuotationById(req, res);
    } catch (error) {
      next(error);
    }
  }
);

userRouter.patch(
  "/quotationStatus/:id", authenticateUser,
  authorizeRole(["admin","user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await quotationController.updateQuotationStatus (req, res);
    } catch (error) {
      next(error);
    }
  }
);



userRouter.put(
  "/quotations/:id",
  authenticateUser,
  authorizeRole(["user"]),
  async (req, res, next) => {
    try {
      await quotationController.updateQuotation(req, res);
    } catch (error) {
      next(error);
    }
  }
);
userRouter.delete(
  "/quotations/:id",
  authenticateUser,
  authorizeRole(["user"]),
  async (req, res, next) => {
    try {
      await quotationController.deleteQuotation(req, res);
    } catch (error) {
      next(error);
    }
  }
);

userRouter.post("/duplicateQuotation/:id", async (req: Request, res: Response, next: NextFunction) => {
  try {
    await quotationController.duplicateQuotation(req, res);
  } catch (error) {
    next(error);
  }
});

userRouter.get(
  "/getquotationbyleadid/:leadId", authenticateUser,
  authorizeRole(["user"]),  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await quotationController.getQuotationByLeadRefNo (req, res);
    } catch (error) {
      next(error);
    }
  }
); 


userRouter.post("/leads", authenticateUser, (req, res, next) =>
  userLeadController.createLead(req, res, next)
);
userRouter.get("/leads", authenticateUser, (req, res, next) =>
  userLeadController.getUserLeads(req, res, next)
);
userRouter.get("/leads/:leadId", authenticateUser, (req, res, next) =>
  userLeadController.getLeadById(req, res, next)
);
userRouter.put("/leads/:leadId", authenticateUser, (req, res, next) =>
  userLeadController.updateLead(req, res, next)
);
userRouter.delete("/leads/:leadId", authenticateUser, (req, res, next) =>
  userLeadController.deleteLead(req, res, next)
);

userRouter.put(
  "/leads/:id/status",
  authenticateUser,
  authorizeRole(["user"]),
  (req, res, next) => {
    userLeadController.updateStatus(req, res, next).catch(next); // Pass 'next'
  }
);

userRouter.put(
  "/leads/:id/stage",
  authenticateUser,
  authorizeRole(["user"]),
  (req, res, next) => {
    userLeadController.updateStage(req, res, next).catch(next); // Pass 'next'
  }
);
// DOCUMENT ROUTES
userRouter.post(
  "/leads/:leadId/documents",
  authenticateUser,
  authorizeRole(["admin", "user", "agent"]),
  documentUpload.single("document"), // Use documentUpload middleware
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await documentController.uploadDocument(req, res, next);
    } catch (error) {
      next(error);
    }
  }
);

userRouter.get(
  "/leads/:leadId/documents",
  authenticateUser,
  authorizeRole(["admin", "user", "agent"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await documentController.getDocumentsForLead(req, res, next);
    } catch (error) {
      next(error);
    }
  }
);

userRouter.delete(
  "/documents/:id",
  authenticateUser,
  authorizeRole(["admin", "agent", "user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await documentController.deleteDocument(req, res, next);
    } catch (error) {
      next(error);
    }
  }
);
// ==============================================

userRouter.post(
  "/agents/createAgent",
  authenticateUser,
  authorizeRole(["user"]),
  agentUpload.single("image"), // Apply multer middleware here
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await agentController.createAgent(req, res, next);
    } catch (error) {
      next(error);
    }
  }
);

userRouter.put(
  "/agents/updateAgent/:id",
  authenticateUser,
  authorizeRole(["user"]),
  agentUpload.single("image"), // Apply multer middleware here
  (req: Request, res: Response, next: NextFunction) =>
    agentController.updateAgent(req, res, next)
);
userRouter.get(
  "/agents",
  (req: Request, res: Response, next: NextFunction) =>
    agentController.getAllAgents(req, res, next)
);

userRouter.delete(
  "/agents/:id",
  authenticateUser,
  authorizeRole(["user"]),
  (req: Request, res: Response, next: NextFunction) => {
    // Call the deleteAgent method from the agentController
    agentController.softDeleteAgent(req, res, next);
    // If you want to implement hard delete, you can call the deleteAgent method instead
    // agentController.deleteAgent(req, res, next);
    // You can also log the ID of the agent being deleted if needed
    console.log("delete", req.params.id);
  }
);

userRouter.put(
  "/agents/restore/:id",
  authenticateUser,
  authorizeRole(["user"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentController.restoreAgent(req, res, next);
  }
);
userRouter.delete(
  "/agents/permenantDelete/:id",
  authenticateUser,
  authorizeRole(["user"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentController.deleteAgent(req, res, next);
  }
);


userRouter.get(
    "/operation",
    authenticateUser, authorizeRole(["admin", "user", "operations"]),
    getOperationDashboardData);

userRouter.get(
    "/top5bookings",
    authenticateUser, authorizeRole(["admin", "user", "operations"]),
    getTop5Bookings);

userRouter.get(
    "/bookingLeadOverview/:timeframe?",
    authenticateUser,
    authorizeRole(["admin", "user", "operations"]),
    getBookingLeadOverview
);

// ==============================================
// BOOKING MANAGEMENT ROUTES
// ==============================================
userRouter.post(
  "/createBooking",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  async (req, res, next) => {
    try {
      await bookingController.createBooking(req, res);
    } catch (error) {
      next(error);
    }
  }
);

userRouter.get(
  "/bookings",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  async (req, res) => {
    try {
      await bookingController.getBookings(req, res);
    } catch (error) {
      console.log(error);
    }
  }
);

userRouter.get(
  "/bookings/:id",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  async (req, res, next) => {
    try {
      await bookingController.getBookingById(req, res);
    } catch (error) {
      next(error);
    }
  }
);

userRouter.patch(
  "/bookings/:id/status",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  (req: Request, res: Response) => {
    bookingController.updateBookingStatus(req, res);
  }
);

userRouter.patch(
  "/bookings/:id/stage",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  (req: Request, res: Response) => {
    bookingController.updateBookingStage(req, res);
  }
);

userRouter.patch(
    "/bookings/updateAccommodation/:bookingId/:accommodationId",
    authenticateUser,
    authorizeRole(["admin","user"]),
    (req: Request, res: Response) => {
        bookingController.updateAccommodationHandler(req, res);
    }
  );

userRouter.put(
    '/bookings/updateItinerary/:bookingId/:itineraryId',
    authenticateUser,
    authorizeRole(["admin","user"]),
    (req: Request, res: Response) => {
        bookingController.updateItinerary(req, res);
    }
);
// ==============================================
// TOUR MANAGEMENT ROUTES
// ==============================================

userRouter.get(
    "/tours",
    (req: Request, res: Response, next: NextFunction) => {
        tourController.getTours(req, res, next).catch(next);
    }
);
userRouter.get(
    "/tours/:id",
    (req: Request, res: Response, next: NextFunction) => {
        tourController.getTourById(req, res, next).catch(next);
    }
);
// ==============================================
// guide MANAGEMENT ROUTES
// ==============================================
userRouter.get("/guides", (req: Request, res: Response, next: NextFunction) => {
    guideController.getGuides(req, res, next);
});
userRouter.get("/getGuide/:id", (req: Request, res: Response, next: NextFunction) => {
    guideController.getGuideById(req, res, next);
});

userRouter.get("/suppliers", async (req, res) => {
    const suppliers = await prisma.suppliers.findMany(); // Or relevant DB fetch
    res.json(suppliers);
});

userRouter.get("/assessments",
  authenticateUser,
  authorizeRole(["user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await userAssessmentController.getAllAssessments(req, res);
    } catch (error) {
      console.error("Error fetching assessments:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
      
  }
)

userRouter.get("/assessments/:id",
  authenticateUser,
  authorizeRole(["user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await userAssessmentController.getAssessmentById(req, res);
    } catch (error) {
      console.error("Error fetching assessment by ID:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);
userRouter.get(
  "/notification",
  authenticateUser,
  authorizeRole(["user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await getUserNotifications(req, res);
    } catch (error) {
      console.error("Error fetching notifications:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }
);







export default userRouter;