import express, { NextFunction, Request, Response } from "express";

// --- Dependency Imports ---
// Login related
import { agentLoginController } from "../controllers/agent/agentLoginController";

// Lead Management Dependencies
import { AgentLeadController } from "../controllers/agent/AgentLeadController";
import { AgentQuotationController } from "../controllers/agent/AgentQuotationController";
import { authenticateUser } from "../middlewares/authMiddleware";
import { authorizeRole } from "../middlewares/roleMiddleware";
import { AgentLeadRepository } from "../repositories/agent/AgentLeadRepository";
import { AgentQuotationRepository } from "../repositories/agent/AgentQuotationRepository";
import { AgentLeadService } from "../services/agent/AgentLeadService";
import { AgentQuotationService } from "../services/agent/AgentQuotationService";

// --- End Dependency Imports ---

const agentRouter = express.Router();

// --- Instantiate Dependencies (if not already done elsewhere) ---
const agentLeadRepository = new AgentLeadRepository();
const agentLeadServiceInstance = new AgentLeadService(agentLeadRepository);
const agentLeadController = new AgentLeadController(agentLeadServiceInstance);

const agentQuotationRepository = new AgentQuotationRepository();
const agentQuotationService = new AgentQuotationService(
  agentQuotationRepository
);
const agentQuotationController = new AgentQuotationController(
  agentQuotationService
);

// --- Route Definitions ---

// Agent Login Route - Public
agentRouter.post("/login", agentLoginController);

// --- Agent Specific Lead Management Routes ---
// These routes are protected and only accessible by authenticated agents

// POST /agents/leads - Create a new lead for the logged-in agent
agentRouter.post(
  "/leads",
  authenticateUser,
  authorizeRole(["agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentLeadController.createLead(req, res, next);
  }
);

// GET /agents/leads - Get all leads for the logged-in agent
agentRouter.get(
  "/leads",
  authenticateUser,
  authorizeRole(["agent"]),
  (req, res, next) => {
    agentLeadController.getAllLeads(req, res, next); // ✅ CORRECTED METHOD NAME
  }
);

agentRouter.get(
  "/leads/:id",
  authenticateUser,
  authorizeRole(["agent"]),
  (req, res, next) => {
    agentLeadController.getLeadById(req, res, next); //  ✅ CORRECTED METHOD NAME
  }
);

// PUT /agents/leads/:leadId - Update a specific lead by ID for the logged-in agent
agentRouter.put(
  "/leads/:leadId",
  authenticateUser,
  authorizeRole(["agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentLeadController.updateLead(req, res, next);
  }
);

// DELETE /agents/leads/:leadId - Delete a specific lead by ID for the logged-in agent
agentRouter.delete(
  "/leads/:leadId",
  authenticateUser,
  authorizeRole(["agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentLeadController.deleteLead(req, res, next);
  }
);

// Quotation routes for agents
agentRouter.post(
  "/quotations",
  authenticateUser,
  authorizeRole(["agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentQuotationController.createQuotation(req, res).catch(next);
  }
);

agentRouter.get(
  "/quotations",
  authenticateUser,
  authorizeRole(["agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentQuotationController.getQuotations(req, res).catch(next);
  }
);

agentRouter.get(
  "/quotations/:id",
  authenticateUser,
  authorizeRole(["agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentQuotationController.getQuotationById(req, res).catch(next);
  }
);

agentRouter.put(
  "/quotations/:id",
  authenticateUser,
  authorizeRole(["agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentQuotationController.updateQuotation(req, res).catch(next);
  }
);

agentRouter.delete(
  "/quotations/:id",
  authenticateUser,
  authorizeRole(["agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentQuotationController.deleteQuotation(req, res).catch(next);
  }
);



export default agentRouter;
