import { PrismaClient, QuotationStatus } from "@prisma/client";
import { NotificationService } from "../../services/common/notificationService";

const prisma = new PrismaClient();

export class QuotationRepository {
  private notificationService: NotificationService;

  constructor() {
    this.notificationService = new NotificationService(); // Initialize once
  }

  // Existing getDuplicateRefNo method (no changes needed here)
  getDuplicateRefNo(originalRefNo: string, existingRefNos: string[]): string {
    // Remove any trailing -D{number} from the original refNo
    const baseRefNo = originalRefNo.replace(/-D\d+$/, "");

    // Find all duplicates of this base refNo
    const duplicates = existingRefNos.filter((ref) =>
      ref.startsWith(`${baseRefNo}-D`)
    );

    // Find highest -D{number}
    const maxDuplicateNumber = duplicates.reduce((max, ref) => {
      const match = ref.match(/-D(\d+)$/);
      const num = match ? parseInt(match[1], 10) : 0;
      return Math.max(max, num);
    }, 0);

    // Return next duplicate refNo
    return `${baseRefNo}-D${maxDuplicateNumber + 1}`;
  }

  async createQuotation(data: any, userid: string) {
    try {
      const parseDate = (d: string | null | undefined) =>
        d ? new Date(d) : null;

      const generateRefNo = async () => {
        const leadId = data.leadRefNo?.replace(/^LD-/, "") ?? "UNKNOWN";
        const baseRef = `QT-${leadId}`;

        // Get only original and revised quotations (exclude duplicates like -D1, -D2)
        // This logic for `generateRefNo` is for revisions/new, not for -D duplicates.
        // It correctly finds the revision count based on QT-LEADID or QT-LEADID-R#
        const existingQuotations = await prisma.quotation.findMany({
          where: {
            leadRefNo: data.leadRefNo,
            userid: userid, // Added userid filter
            NOT: {
              refNo: {
                endsWith: '-D1', // This condition is too specific, needs to be more general
              },
            },
          },
          select: {
            refNo: true,
          },
        });

        const revisionCount = existingQuotations.filter(q =>
          q.refNo === baseRef || (q.refNo.startsWith(`${baseRef}-R`) && !q.refNo.includes('-D'))
        ).length;


        return revisionCount === 0 ? baseRef : `${baseRef}-R${revisionCount + 1}`;
      };

      let refNo: string;
      if (data.refNo?.includes("-D")) {
        refNo = data.refNo; // This path won't be used by the duplicate function we are creating
      } else if (data.refNo) {
        // This implies 'data.refNo' is provided but doesn't have '-D', so it's a revision request
        refNo = await generateRefNo(); // Revision request
        console.log("Generated revision refNo:", refNo);
      } else {
        refNo = await generateRefNo(); // Fresh quotation (no refNo provided)
        console.log("Generated new refNo:", refNo);
      }

      // const newStage = "quotation_created"; //No lead update at user end

      const quotation = await prisma.quotation.create({
        data: {
          refNo,
          leadRefNo: data.leadRefNo,
          leadAgent: data.leadAgent ?? null,
          status: data.status ?? null,
          guestName: data.guestName ?? null,
          remarks: data.remarks ?? null,
          currency: data.currency ?? null,
          totalBeforeMarkup: data.totalBeforeMarkup ?? null,
          totalAfterMarkup: data.totalAfterMarkup ?? null,
          travelDates: data.travelDates,
          userid: userid, // Associate with user
          agentId: data.agentId ?? null,

          accommodations: {
            create: (data.accommodations ?? []).map((acc: any) => ({
              hotel: acc.hotel ?? null,
              hotelAddress: acc.hotelAddress ?? null,
              supplierId: acc.supplierId ?? null,
              supplierName: acc.supplierName ?? null,
              supplierContact: acc.supplierContact ?? null,
              room: acc.room ?? null,
              travelDate: parseDate(acc.travelDate),
              endDate: parseDate(acc.endDate),
              nights: acc.nights ?? null,
              surchargeNights: acc.surchargeNights ?? null,
              quantity: acc.quantity ?? null,
              totalPrice: acc.totalPrice ?? null,
              totalPriceAfterMarkup: acc.totalPriceAfterMarkup ?? null,
              priceDetailsAccommodation: {
                create: (acc.priceDetails ?? []).map((pd: any) => ({
                  roomId: pd.roomId ?? null,
                  category: pd.category ?? null,
                  pax: pd.pax ?? null,
                  price: pd.price ?? null,
                  priceAfterMarkup: pd.priceAfterMarkup ?? null,
                  costPrice: pd.costPrice ?? null,
                  optionalPrice: pd.optionalPrice ?? null,
                  quantity: pd.quantity ?? null,
                  totalPrice: pd.totalPrice ?? null,
                  totalPriceAfterMarkup: pd.totalPriceAfterMarkup ?? null,
                })),
              },
            })),
          },

          itineraries: {
            create: (data.itineraries ?? []).map((itin: any) => ({
              date: parseDate(itin.date),
              label: itin.label ?? null,
              category: itin.category ?? null,
              supplierId: itin.supplierId ?? null,
              supplierName: itin.supplierName ?? null,
              supplierContact: itin.supplierContact ?? null,
              totalPriceAfterMarkup: itin.totalPriceAfterMarkup ?? null,
              priceDetailsList: {
                create: (itin.priceDetails ?? []).map((pd: any) => ({
                  category: pd.category ?? null,
                  price: pd.price ?? null,
                  priceAfterMarkup: pd.priceAfterMarkup ?? null,
                  quantity: pd.quantity ?? null,
                  totalPrice: pd.totalPrice ?? null,
                  totalPriceAfterMarkup: pd.totalPriceAfterMarkup ?? null,
                })),
              },
              guideId: itin.guideId ?? null,
            })),
          },

          tours: {
            connect: (data.tours ?? []).map((tourId: string) => ({
              id: tourId,
            })),
          },
        },
        include: {
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          tours: true,
        },
      });

      const user = await prisma.user.findUnique({
        where: { id: userid },
        select: { name: true },
      });

      const userName = user?.name || "Unknown User";

      const lead = await prisma.lead.findUnique({
        where: { leadNo: data.leadRefNo },
      });

      if (lead) {
        const newStage = "quotation_created"; // Define the new stage value
        await prisma.lead.update({
          where: { id: lead.id },
          data: { stage: newStage },
        });

        await prisma.followUp.create({
          data: {
            leadId: lead.id,
            oldStage: lead.stage,
            newStage: newStage,
            notes: "Quotation created",
            userName: userName ?? null,
          },
        });
      }

      // ✅ Send notification
      await this.notificationService.notifyAdmin({
        type: "quotation_created",
        title: "New Quotation Created",
        message: `New quotation created by ${userName}.\nLead Reference No: ${data.leadRefNo || "N/A"
          }\nGuest Name: ${data.guestName || "Guest Name"}.`,
        entityType: "quotation",
        entityId: quotation.id,
        // Change as needed
      });

      console.log("Quotation created in repo: ******************", quotation.refNo);
      return quotation;
    } catch (error) {
      console.error("Database error while creating quotation:", error);
      throw new Error("Failed to create quotation. Please try again later.");
    }
  }

  // --- NEW DUPLICATION METHOD ---
  async duplicateQuotation(id: string, userid: string) {
    try {
      const originalQuotation = await prisma.quotation.findUnique({
        where: { id, userid: userid }, // Added userid filter
        include: {
          accommodations: {
            include: { priceDetailsAccommodation: true },
          },
          itineraries: {
            include: { priceDetailsList: true },
          },
          tours: true, // Make sure to include tours for copying
        },
      });

      if (!originalQuotation) {
        throw new Error("Original quotation not found for duplication or user doesn't own it.");
      }

      // 1. Get all refNos for this lead to determine the next duplicate number
      const allQuotationsForLead = await prisma.quotation.findMany({
        where: { leadRefNo: originalQuotation.leadRefNo, userid: userid }, // Added userid filter
        select: { refNo: true },
      });
      const existingRefNosForLead = allQuotationsForLead.map((q) => q.refNo);

      // The base for duplication should be the original refNo without any -D suffix,
      // as new duplicates will be appended to this base.
      const baseRefForDuplication = originalQuotation.refNo.replace(
        /-D\d+$/,
        ""
      );
      const newRefNo = this.getDuplicateRefNo(
        baseRefForDuplication,
        existingRefNosForLead
      );

      // 2. Prepare data for the new quotation
      // Dates from Prisma are already Date objects, so parseDate needs adjustment for this context
      const parseDate = (d: Date | null | undefined) => (d ? d : null);

      const duplicateData: any = {
        refNo: newRefNo,
        leadRefNo: originalQuotation.leadRefNo,
        leadAgent: originalQuotation.leadAgent,
        status: QuotationStatus.Draft, // Set duplicated quotation status to 'Draft'
        guestName: originalQuotation.guestName,
        remarks: `Duplicate of ${originalQuotation.refNo} (created by user ${userid} at ${new Date().toISOString()})`, // Add a remark
        currency: originalQuotation.currency,
        totalBeforeMarkup: originalQuotation.totalBeforeMarkup,
        totalAfterMarkup: originalQuotation.totalAfterMarkup,
        travelDates: originalQuotation.travelDates,
        userid: userid,
        agentId: originalQuotation.agentId,
        // createdAt and updatedAt will be automatically set by Prisma for the new record
      };

      // Handle nested relations for creation
      duplicateData.accommodations = {
        create: originalQuotation.accommodations.map((acc) => ({
          hotel: acc.hotel,
          hotelAddress: acc.hotelAddress,
          supplierId: acc.supplierId,
          supplierName: acc.supplierName,
          supplierContact: acc.supplierContact,
          room: acc.room,
          travelDate: parseDate(acc.travelDate),
          endDate: parseDate(acc.endDate),
          nights: acc.nights,
          surchargeNights: acc.surchargeNights,
          quantity: acc.quantity,
          totalPrice: acc.totalPrice,
          totalPriceAfterMarkup: acc.totalPriceAfterMarkup,
          priceDetailsAccommodation: {
            create: acc.priceDetailsAccommodation.map((pd) => ({
              roomId: pd.roomId,
              category: pd.category,
              pax: pd.pax,
              price: pd.price,
              priceAfterMarkup: pd.priceAfterMarkup,
              costPrice: pd.costPrice,
              optionalPrice: pd.optionalPrice,
              quantity: pd.quantity,
              totalPrice: pd.totalPrice,
              totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
            })),
          },
        })),
      };

      duplicateData.itineraries = {
        create: originalQuotation.itineraries.map((itin) => ({
          date: parseDate(itin.date),
          label: itin.label,
          category: itin.category,
          supplierId: itin.supplierId,
          supplierName: itin.supplierName,
          supplierContact: itin.supplierContact,
          totalPriceAfterMarkup: itin.totalPriceAfterMarkup,
          priceDetailsList: {
            create: itin.priceDetailsList.map((pd) => ({
              category: pd.category,
              price: pd.price,
              priceAfterMarkup: pd.priceAfterMarkup,
              quantity: pd.quantity,
              totalPrice: pd.totalPrice,
              totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
            })),
          },
          guideId: itin.guideId,
        })),
      };

      // For many-to-many relationships (like tours), use `connect` to link existing tours
      duplicateData.tours = {
        connect: originalQuotation.tours.map((tour) => ({ id: tour.id })),
      };

      const duplicatedQuotation = await prisma.quotation.create({
        data: duplicateData,
        include: {
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          tours: true,
        },
      });

      console.log("Quotation duplicated in repo:", duplicatedQuotation.refNo);
      return duplicatedQuotation;
    } catch (error) {
      if (error instanceof Error) {
        console.error("Error duplicating quotation:", error.message);
      } else {
        console.error("Error duplicating quotation:", error);
      }
      throw new Error("Failed to duplicate quotation");
    }
  }



  async getQuotationByLeadRefNo(leadRefNo: string) {
    try {
      const quotation = await prisma.quotation.findMany({
        where: { leadRefNo}, // Added userid filter
        include: {
          accommodations: {
            include: { priceDetailsAccommodation: true },
          },
          itineraries: {
            include: { priceDetailsList: true }
          },
          tours: true,
          lead: {
            include: {
              Agent: true, // Note: Capital 'A' to match your schema
              User: true, // Include user details
            },
          },
        },
      });

      if (!quotation) {
        throw new Error("Quotation not found");
      }
      console.log("Quotation in repo", quotation);
      return quotation;
    } catch (error) {
      console.error("Error fetching quotation:", error);
      throw error;
    }
  }
  async getQuotationById(id: string, userid: string) {
    try {
      const quotation = await prisma.quotation.findUnique({
        where: { id, userid: userid }, // Added userid filter
        include: {
          accommodations: {
            include: { priceDetailsAccommodation: true },
          },
          itineraries: {
            include: { priceDetailsList: true },
          },
          tours: true,
          lead: {
            include: {
              Agent: true, // Note: Capital 'A' to match your schema
            },
          },
        },
      });

      if (!quotation) {
        throw new Error("Quotation not found");
      }
      console.log("Quotation in repo", quotation);
      return quotation;
    } catch (error) {
      console.error("Error fetching quotation:", error);
      throw error;
    }
  }

  async getAllQuotations(userid: string) {
    try {
      const quotations = await prisma.quotation.findMany({
        where: { userid: userid }, // Added userid filter
        include: {
          accommodations: {
            include: { priceDetailsAccommodation: true },
          },
          itineraries: {
            include: { priceDetailsList: true }, // Fix incorrect reference
          },
          tours: true,
        },
      });

      return quotations;
    } catch (error) {
      console.error("Error fetching quotations:", error);
      throw new Error("Could not fetch quotations");
    }
  }

async updateQuotation(id: string, data: any, userid: string) {
  try {
    const parseDate = (d: string | null | undefined) =>
      d ? new Date(d) : null;

    // Delete nested children first.  Using `deleteMany` with relations is generally more efficient.
    await prisma.priceDetailsAccommodation.deleteMany({
      where: { accommodation: { quotationId: id } },
    });
    await prisma.accommodation.deleteMany({ where: { quotationId: id } });

    await prisma.priceDetailItinerary.deleteMany({
      where: { itinerary: { quotationId: id } },
    });
    await prisma.itinerary.deleteMany({ where: { quotationId: id } });

    // Disconnect tours before updating.  Important for proper updates
    await prisma.quotation.update({
      where: { id, userid: userid },
      data: {
        tours: {
          set: [],
        },
      },
    });


    const updateData: any = {
      leadRefNo: data.leadRefNo,
      leadAgent: data.leadAgent ?? null,
      status: data.status ?? null,
      guestName: data.guestName ?? null,
      remarks: data.remarks ?? null,
      currency: data.currency ?? null,
      totalBeforeMarkup: data.totalBeforeMarkup ?? null,
      totalAfterMarkup: data.totalAfterMarkup ?? null,
      travelDates: data.travelDates,
      agentId: data.agentId ?? null,
    };


    if (data.accommodations) {
      updateData.accommodations = {
        create: data.accommodations.map((acc: any) => ({
          hotel: acc.hotel ?? null,
          hotelAddress: acc.hotelAddress ?? null,
          supplierId: acc.supplierId ?? null,
          supplierName: acc.supplierName ?? null,
          supplierContact: acc.supplierContact ?? null,
          room: acc.room ?? null,
          travelDate: parseDate(acc.travelDate),
          endDate: parseDate(acc.endDate),
          nights: acc.nights ?? null,
          surchargeNights: acc.surchargeNights ?? null,
          quantity: acc.quantity ?? null,
          totalPrice: acc.totalPrice ?? null,
          totalPriceAfterMarkup: acc.totalPriceAfterMarkup ?? null,
          priceDetailsAccommodation: {
            create: (acc.priceDetails ?? []).map((pd: any) => ({
              roomId: pd.roomId ?? null,
              category: pd.category ?? null,
              pax: pd.pax ?? null,
              price: pd.price ?? null,
              priceAfterMarkup: pd.priceAfterMarkup ?? null,
              costPrice: pd.costPrice ?? null,
              optionalPrice: pd.optionalPrice ?? null,
              quantity: pd.quantity ?? null,
              totalPrice: pd.totalPrice ?? null,
              totalPriceAfterMarkup: pd.totalPriceAfterMarkup ?? null,
            })),
          },
        })),
      };
    }

    if (data.itineraries) {
      updateData.itineraries = {
        create: data.itineraries.map((itin: any) => ({
          date: parseDate(itin.date),
          label: itin.label ?? null,
          category: itin.category ?? null,
          supplierId: itin.supplierId ?? null,
          supplierName: itin.supplierName ?? null,
          supplierContact: itin.supplierContact ?? null,
          totalPriceAfterMarkup: itin.totalPriceAfterMarkup ?? null,
          priceDetailsList: {
            create: (itin.priceDetails ?? []).map((pd: any) => ({
              category: pd.category ?? null,
              price: pd.price ?? null,
              priceAfterMarkup: pd.priceAfterMarkup ?? null,
              quantity: pd.quantity ?? null,
              totalPrice: pd.totalPrice ?? null,
              totalPriceAfterMarkup: pd.totalPriceAfterMarkup ?? null,
            })),
          },
          guideId: itin.guideId ?? null,
        })),
      };
    }

    if (data.tours) {
      updateData.tours = {
        connect: data.tours.map((tourId: string) => ({ id: tourId })),
      };
    }



    const updatedQuotation = await prisma.quotation.update({
      where: { id, userid: userid },
      data: updateData,
      include: {
        accommodations: { include: { priceDetailsAccommodation: true } },
        itineraries: { include: { priceDetailsList: true } },
        tours: true,
      },
    });


     // Simulate the admin's updateQuotationStatus behavior for 'Accepted' status
     if (data.status === 'Accepted') { // Use data.status instead of updatedQuotation.status as it may not be updated yet

      const leadRefNo = updatedQuotation.leadRefNo;

      if (leadRefNo) {

        const lead = await prisma.lead.findUnique({ // Need to fetch lead data
          where: { leadNo: leadRefNo },
        });

        if (lead) { // Check if the lead exists
        await prisma.lead.update({
          where: { leadNo: leadRefNo },
          data: {
            stage: 'quotation_accepted',
            followUps: {
              create: {
                oldStage: lead.stage, // Use the fetched lead's stage
                newStage: 'quotation_accepted',
                notes: `Quotation ${updatedQuotation.refNo} accepted.`, // Access refNo after quotation update
                userName: userid, // Or the actual user name
              },
            },
          },
        });
       } else {
        console.warn(`Lead with leadNo ${leadRefNo} not found.  Skipping lead update.`);
       }
      }  else {
        console.warn('leadRefNo is null or undefined.  Skipping lead update.');
      }
    }


    console.log("Updated Quotation", updatedQuotation);

    return updatedQuotation;
  } catch (error) {
    console.error("Error updating quotation:", error);
    throw new Error("Failed to update quotation");
  }
}

  async deleteQuotation(id: string, userid: string) {
    console.log("ID in repository:", id); // Debugging

    const existingQuotation = await prisma.quotation.findUnique({
      where: { id, userid: userid }, // Added userid filter
    });

    if (!existingQuotation) {
      throw new Error("Quotation not found");
    }

    return await prisma.quotation.delete({
      where: { id },
    });
  }

  async updateQuotationStatus(id: string, status: QuotationStatus, userid: string) {
    try {



    // Find the user's name (optional - for follow-up notes)
        const user = await prisma.user.findUnique({
          where: { id: userid },
          select: { name: true }
        });
        const userName = user?.name || "Unknown User";


      const updatedQuotation = await prisma.quotation.update({
        where: { id, userid: userid }, // <--- ADDED userid to where clause
        data: { status },
        include: {
          lead: true, // so we can access lead data
          accommodations: {
            include: { priceDetailsAccommodation: true },
          },
          itineraries: {
            include: { priceDetailsList: true },
          },
          tours: true,
        },
      });

      if (!updatedQuotation) throw new Error("Quotation not found");

      // If status is 'Accepted', also update Lead and create FollowUp
      if (status === 'Accepted') {

    

        const leadUpdate = await prisma.lead.update({
          where: { leadNo: updatedQuotation.leadRefNo },
          data: {
            stage: 'quotation_accepted',
            followUps: {
              create: {
                oldStage: updatedQuotation.lead.stage,
                newStage: 'quotation_accepted',
                notes: `Quotation ${updatedQuotation.refNo} accepted by ${userName}.`, // Use the actual username here
                userName: userName, // Use the actual username here
              },
            },
          },
        });
      }

         await this.notificationService.notifyAdmin({
          type: "quotation_accepted",
          title: "Quotation Accepted",
          message: `Quotation ${updatedQuotation.refNo} has been accepted by ${userName}.`,
          entityType: "quotation",
          entityId: updatedQuotation.id,
        });
      

      return updatedQuotation;
    } catch (error) {
      console.error("Error updating quotation status:", error);
      throw error;
    }
  }

  async getAcceptedQuotationsByLeadId(leadId: string, userid: string) {
    try {
      const lead = await prisma.lead.findUnique({
        where: { id: leadId },
        include: {
          Quotation: {
            where: { status: 'Accepted', userid: userid }, // Added userid filter
            include: {
              accommodations: true,
              itineraries: true,
              tours: true,
            },
          },
        },
      });

      if (!lead) throw new Error("Lead not found");
      return lead;
    } catch (error) {
      console.error("Error fetching accepted quotations:", error);
      throw error;
    }
  }
}