// In OperationDashboardRepository.ts

import prisma from "../../config/prismaClient";
import { LeadStage, LeadStatus } from "@prisma/client";
import { subDays, subWeeks, subMonths, subYears, format, startOfDay, endOfDay } from 'date-fns';

export class OperationDashboardRepository {
    async getDashboardData(userId: string) {
        try {

            const assignedBookingsCount = await prisma.booking.count({
                where: {
                    lead: {
                        assignedOperationId: userId,  // Only bookings assigned to the user
                    }
                }
            });


            // Calculate total **quote price** (total Revenue)
            const totalQuotePriceAggregate = await prisma.booking.aggregate({
                where: {
                    lead: {
                        assignedOperationId: userId, // Only bookings assigned to the user
                    }
                },
                _sum: {
                    quotePrice: true,
                },
            });
            const totalQuotePrice = totalQuotePriceAggregate._sum.quotePrice || 0;

            // Calculate total **actual cost** (total Cost) from booking
            const totalActualCostAggregate = await prisma.booking.aggregate({
                where: {
                    lead: {
                        assignedOperationId: userId, // Only bookings assigned to the user
                    }
                },
                _sum: {
                    actualCost: true,
                },
            });
            const totalActualCost = totalActualCostAggregate._sum.actualCost || 0;

           // Calculate total **actual price** based on accommodations and itineraries
            let totalActualAccomodationPrice = 0;
            let totalActualItineraryPrice = 0;

             const bookings = await prisma.booking.findMany({
                where: {
                    lead: {
                        assignedOperationId: userId,
                    },
                },
                include: {
                    accommodations: true,
                    itineraries: true,
                },
            });

            bookings.forEach(booking => {
                booking.accommodations.forEach(accommodation => {
                  if (accommodation.actualPrice) {
                       totalActualAccomodationPrice += accommodation.actualPrice.toNumber();
                    }
                });
                booking.itineraries.forEach(itinerary => {
                   if (itinerary.actualPrice) {
                       totalActualItineraryPrice += itinerary.actualPrice.toNumber();
                    }
                });
            });

            const totalActualOverallPrice = totalActualAccomodationPrice + totalActualItineraryPrice;


            // Calculate total profit (quotePrice - actualCost)
            const totalProfit = totalQuotePrice - totalActualCost;

            const latestBooking = await prisma.booking.findFirst({
                where: {
                    lead: {
                        assignedOperationId: userId,  // Only bookings assigned to the user
                    }
                },
                orderBy: { updatedAt: 'desc' },
                include: { lead: true }
            });


            return {
                totalBookings: assignedBookingsCount,
                totalQuotePrice: totalQuotePrice, // Total Revenue (quotePrice)
                totalActualCost: totalActualCost, // Total Cost (actualCost)
                totalProfit: totalProfit, // Total Profit (quotePrice - actualCost)
                latestBooking,
                totalActualOverallPrice
            };
        } catch (error: any) {
            console.error("Repository Error:", error);
            throw new Error("Failed to retrieve Operation dashboard data");
        }
    }

    async getTopBookingsByProfit(userId: string, limit: number = 5) {
        try {
            const topBookings = await prisma.booking.findMany({
                where: {
                    lead: {
                        assignedOperationId: userId,  // Only bookings assigned to the user
                    }
                },
                orderBy: {
                    profit: 'desc'
                },
                take: limit,
                select: {
                    id: true,
                    reference: true,
                    actualCost: true,
                    quotePrice: true,
                    profit: true,
                    lead: {
                        select: {
                            agentName: true,
                            id: true,
                            assignedOperationId: true
                        }
                    }
                },
            });

            // Calculate profit for each booking (if not already correct)
            const bookingsWithCalculatedProfit = topBookings.map(booking => ({
                ...booking,
                profit: booking.quotePrice - booking.actualCost,  // Calculate profit
            }));

            return bookingsWithCalculatedProfit;
        } catch (error: any) {
            console.error("Repository Error:", error);
            throw new Error("Failed to retrieve Top 5 Bookings");
        }
    }

async  getBookingLeadOverview(timeframe: string = "weekly", userId: string) {
     try {
        let startDate: Date;
        let dateFormat: string;
        let endDate = new Date(); // Default end date is now

        const today = new Date(); // Use today as the reference point

        switch (timeframe) {
            case 'today':
                startDate = startOfDay(today); //start of the day
                endDate = endOfDay(today); // end of the day
                dateFormat = 'HH:mm'; //Format time
                break;
            case 'weekly':
                startDate = subWeeks(today, 1);
                dateFormat = 'yyyy-MM-dd';
                break;
            case 'monthly':
                startDate = subMonths(today, 1);
                dateFormat = 'yyyy-MM';
                break;
            case 'yearly':
                startDate = subYears(today, 1);
                dateFormat = 'yyyy';
                break;
            default:
                startDate = subWeeks(today, 1);
                dateFormat = 'yyyy-MM-dd';
                break;
        }

        // Fetch bookings instead of leads, and group by lead ID
        const bookings = await prisma.booking.findMany({
            where: {
                createdAt: {
                    gte: startDate,
                    lte: endDate,
                },
                lead: {
                    assignedOperationId: userId, // Only bookings related to leads assigned to the user
                },
            },
            select: {
                createdAt: true,
                lead: {
                    select: {
                        id: true, // Lead ID for de-duplication
                        status: true, //Lead Status
                    }
                },
                // Add other booking fields you might need for categorization/analysis here
            },
        });


        const leadData: { [date: string]: { leads: Set<string>; booked: number; cancelled: number } } = {};

        bookings.forEach(booking => {
            if (!booking.lead) return; // Skip bookings without associated leads (shouldn't happen but good to check)

            const leadId = booking.lead.id;
            const date = format(booking.createdAt, dateFormat);

            if (!leadData[date]) {
                leadData[date] = {
                    leads: new Set<string>(), // Use a Set to track unique lead IDs
                    booked: 0,
                    cancelled: 0,
                };
            }

            leadData[date].leads.add(leadId); // Add the lead ID to the set

            if (booking.lead.status === LeadStatus.active) {
                leadData[date].booked++;
            } else if (booking.lead.status === LeadStatus.closed) {
                leadData[date].cancelled++;
            }
        });

        const chartData = Object.entries(leadData).map(([date, data]) => ({
            date,
            leads: data.leads.size, // Count the unique lead IDs in the set
            booked: data.booked,
            cancelled: data.cancelled,
        }));

        return chartData;
    } catch (error: any) {
        console.error("Error fetching booking leads data:", error);
        throw new Error("Failed to retrieve booking leads data");
    }
    }
}

