import prisma from "../../config/prismaClient";
import { subDays, subWeeks, subMonths, format, startOfDay, endOfDay, isToday } from 'date-fns';
import { LeadStage } from "@prisma/client";

export class AssessmentDashboardRepository {

    private completedStages: LeadStage[] = [LeadStage.trip_completed, LeadStage.feedback_collected];  // THE SOURCE OF TRUTH


    async getDashboardData(userId: string) {
        try {
            const totalAssessments = await prisma.lead.count({
                where: {
                    assignedAssessmentId: userId,
                },
            });

            const completedAssessments = await prisma.lead.count({
                where: {
                    assignedAssessmentId: userId,
                    stage: { in: this.completedStages },
                },
            });

            // Calculate upcoming assessments (assigned to assessment with travelDateFrom >= today)
            const upcomingAssessments = await prisma.lead.count({
                where: {
                    assignedAssessmentId: userId,
                    stage: 'assigned_to_assessment', // Use the string value directly
                    travelDateFrom: {
                        gte: new Date() // Only count assessments with travelDateFrom in the future (or today)
                    },
                      travelDateTo: {
                        gte:new Date()
                    }
                },
            });

            // Return the total assessments and completed assessments

            return {
                totalAssessments,
                completedAssessments,
                upcomingAssessments, // Include upcoming assessments in the return
            };
        } catch (error: any) {
            console.error("Repository Error:", error);
            throw new Error("Failed to retrieve Assessment dashboard data");
        }
    }

    private async getCountsPerDay(
        tableName: 'lead',
        userId: string,
        n = 7,
        timeframe: string = 'weekly'
    ) {
        const dates = this.getLastNDates(n);
        const dateFormat = this.getDateFormat(timeframe);
        const todayFormat = 'dd-MM-yyyy';

        const results = await prisma.lead.findMany({

            where: {
                assignedAssessmentId: userId,
                stage: { in: this.completedStages }, // Important: Only completed stages
                assignedAssessmentOn: { // Use createdOn to find when it was completed.
                    gte: dates[0],
                    lte: endOfDay(new Date()),
                },
            },
            select: {
                assignedAssessmentOn: true, // Select the updated at field
                stage: true,
            },
        });

        // Group the results by formatted date.
        const groupedResults: Record<string, number> = results.reduce((acc: Record<string, number>, curr) => {
            if (!curr.assignedAssessmentOn) {
                return acc;
            }
            const isCreatedToday = isToday(curr.assignedAssessmentOn);

            const dateKey = isCreatedToday
                ? format(curr.assignedAssessmentOn, todayFormat)
                : format(curr.assignedAssessmentOn, dateFormat);

            acc[dateKey] = (acc[dateKey] || 0) + 1;
            return acc;
        }, {});

        // Map the results to the expected format.
        const counts = dates.map(date => {
            const isTodayDate = isToday(date);
            const dateKey = isTodayDate ? format(date, todayFormat) : format(date, dateFormat);

            const completed = groupedResults[dateKey] || 0;

            return {
                leads: 0, //Leads not available here , should be 0
                completed: completed,
            };
        });

        return counts;
    }

    private getDateFormat(timeframe: string): string {
        switch (timeframe) {
            case 'today':
                return 'HH:mm';
            case 'weekly':
                return 'dd-MM-yyyy';
            case 'monthly':
                return 'MM-yyyy';
            case 'yearly':
                return 'yyyy';
            default:
                return 'dd-MM-yyyy';
        }
    }

    private getLastNDates(n: number) {
        const dates = [];
        for (let i = n - 1; i >= 0; i--) {
            dates.push(startOfDay(subDays(new Date(), i)));
        }
        return dates;
    }

    async getAssessmentLeadOverview(timeframe: string, userId: string) {
        try {

            let startDate: Date;
            let endDate: Date = new Date();
            let dateFormat: string;
            const today = new Date();

            // Trim and lowercase the timeframe to handle potential whitespace or casing issues
            const trimmedTimeframe = timeframe.trim().toLowerCase();

            switch (trimmedTimeframe) {
                case 'today':
                    startDate = startOfDay(today);
                    endDate = endOfDay(today);
                    dateFormat = 'HH:mm';
                    break;
                case 'weekly':
                    startDate = subWeeks(today, 1);
                    dateFormat = 'dd-MM-yyyy';
                    break;
                case 'monthly':
                    startDate = subMonths(today, 1);
                    dateFormat = 'MM-yyyy';
                    break;
                case 'yearly':
                    startDate = new Date(today.getFullYear(), 0, 1); // January 1st of the current year
                    startDate = startOfDay(startDate);
                    endDate = endOfDay(today); // End of today
                    dateFormat = 'yyyy-MM-dd';
                    break;
                default:
                    startDate = subWeeks(today, 1);
                    endDate = today; //setting to today prevent errors
                    dateFormat = 'dd-MM-yyyy';
                    break;
            }

            console.log("Timeframe:", trimmedTimeframe);  // Log the trimmed timeframe
            console.log("Query Start Date:", startDate.toISOString());
            console.log("Query End Date:", endDate.toISOString());


            const leads = await prisma.lead.findMany({
                where: {
                    assignedAssessmentId: userId,
                    assignedAssessmentOn: {
                        gte: startDate,
                        lte: endDate,
                    },
                },
                select: {
                    stage: true,
                    assignedAssessmentOn: true,
                    travelDateFrom: true,
                },
            });


            console.log("Number of leads found:", leads.length);

            let totalLeads = 0;
            let totalActiveLeads = 0;
            let totalClosedLeads = 0;

            // Calculate the total, active, and closed leads within the time frame
            totalLeads = leads.length;

            leads.forEach(lead => {
                if (this.completedStages.includes(lead.stage)) {
                    totalClosedLeads++;
                } else {
                    totalActiveLeads++;
                }
            });



            // Refactor leadData calculation for clarity
            const leadData: Record<string, { leads: number; active: number; closed: number }> = {};

            leads.forEach(lead => {
                if (!lead.assignedAssessmentOn) return;

                const date = format(lead.assignedAssessmentOn, dateFormat);

                leadData[date] = leadData[date] || { leads: 0, active: 0, closed: 0 };
                leadData[date].leads++;

                // Determine active/closed status directly based on the stage
                if (this.completedStages.includes(lead.stage)) {
                    leadData[date].closed++;
                } else {
                    leadData[date].active++;
                }
            });

            const chartData = Object.entries(leadData).map(([date, data]) => ({
                date,
                leads: data.leads,
                active: data.active,
                closed: data.closed,
            }));

            // Recalculate totalAssessments and completedAssessments based on leads fetched in this function
            const totalAssessments = leads.length; // All leads in this context are assigned.

            const completedAssessments = totalClosedLeads; // This is the actual completed leads.

            const leadsData = await this.getCountsPerDay('lead', userId, 7, timeframe); // Pass timeframe to getCountsPerDay
            return {
                totalAssessments: totalLeads, // This should be all the leads assigned in timeframe
                completedAssessments: totalClosedLeads, // This should be the closed leads from total leads
                leadOverview: chartData,
                activeAssessment: totalActiveLeads,
                dataSeries: {
                    leads: leadsData.map(item => item.leads),  // Extract leads
                    completed: leadsData.map(item => item.completed),// Extract completed
                },
            };
        } catch (error: any) {
            console.error("Error fetching assessment lead overview data:", error);
            return {
                totalAssessments: 0,
                completedAssessments: 0,
                leadOverview: [],
                activeAssessment: 0,
                dataSeries: {
                    leads: [],
                    completed: [],
                },
            };

        }
    }
}