// backend/src/repositories/agent/agentRepository.ts

import { Prisma } from "@prisma/client";
// Make sure this import path is correct for your interface file
import { nanoid } from "nanoid";
import prisma from "../../config/prismaClient";
import { Agent, AgentCredentials } from "../../interfaces/Agent";

export class AgentRepository {

    async getAgentByEmail(email: string): Promise<Agent | null> {
        try {
            return await prisma.agent.findUnique({
                where: { email: email },
            });
        } catch (error) {
            console.error("Error finding agent by email:", error);
            return null;
        }
    }

    async getAgentByPhone(phone: string): Promise<Agent | null> {
        try {
            return await prisma.agent.findUnique({
                where: { phone: phone },
            });
        } catch (error) {
            console.error("Error finding agent by phone:", error);
            return null;
        }
    }

    async getAgentByAgentCode(agentCode: string): Promise<Agent | null> {
        try {
            return await prisma.agent.findUnique({
                where: { agentCode: agentCode },
            });
        } catch (error) {
            console.error("Error finding agent by agent code:", error);
            return null;
        }
    }

    public async createAgent(
        agentData: Agent,
        credentialsData?: AgentCredentials
    ): Promise<Agent> {
        try {
            // ✅ Create new agent
            const newAgent = await prisma.agent.create({
                data: {
                    agentCode: agentData.agentCode,
                    companyName: agentData.companyName,
                    contactPerson: agentData.contactPerson,
                    designation: agentData.designation,
                    companyAddress: agentData.companyAddress,
                    state: agentData.state,
                    email: agentData.email,
                    phone: agentData.phone,
                    markup: agentData.markup,
                    agentPrivilege: agentData.agentPrivilege,
                    deleted: agentData.deleted ?? false,
                    credentials: credentialsData?.username
                        ? {
                            create: {
                                id: credentialsData.id || nanoid(),
                                username: credentialsData.username,
                                password: credentialsData.password,
                                filename: credentialsData.filename || null,
                            },
                        }
                        : undefined,
                },
                include: { credentials: true },
            });

            return {
                ...newAgent,
                agentInfo: agentData.agentInfo || null,
            } as Agent;
        } catch (error) {
            console.error("❌ Error creating agent in repository:", error);
            throw error;
        }
    }

    public async updateAgent(
        agentId: string,
        agentData: Partial<Agent>,
        credentialsData?: Partial<AgentCredentials>
    ): Promise<Agent | null> {
        try {
            const existingAgent = await prisma.agent.findUnique({
                where: { id: agentId },
                select: {
                    credentialsId: true,
                    email: true,
                    agentCode: true,
                    phone: true,
                },
            });

            if (!existingAgent) {
                console.warn(`Agent with ID ${agentId} not found for update.`);
                return null;
            }

            // ✅ Step 2: Construct update payload
            const updatePayload: Prisma.AgentUpdateInput = {
                agentCode: agentData.agentCode,
                companyName: agentData.companyName,
                contactPerson: agentData.contactPerson,
                designation: agentData.designation,
                companyAddress: agentData.companyAddress,
                state: agentData.state,
                email: agentData.email,
                phone: agentData.phone,
                markup: agentData.markup,
                agentPrivilege: agentData.agentPrivilege,
                deleted: agentData.deleted,
            };

            if (credentialsData) {
                if (existingAgent.credentialsId) {

                  const credentialUpdatePayload: Prisma.AgentCredentialsUpdateInput = {};

                  if (credentialsData.username !== undefined) {
                    credentialUpdatePayload.username = credentialsData.username;
                  }
                  if (credentialsData.password !== undefined) {
                    credentialUpdatePayload.password = credentialsData.password;
                  }
                    credentialUpdatePayload.filename = credentialsData.filename;

                    updatePayload.credentials = {
                        update: credentialUpdatePayload
                    };
                } else {
                    updatePayload.credentials = {
                        create: {
                            id: credentialsData.id || nanoid(),
                            username: credentialsData.username,
                            password: credentialsData.password,
                            filename: credentialsData.filename,
                        },
                    };
                }
            }

            // ✅ Step 3: Perform update
            const updatedAgent = await prisma.agent.update({
                where: { id: agentId },
                data: updatePayload,
                include: { credentials: true },
            });

            return {
                ...updatedAgent,
                agentInfo: agentData.agentInfo || null,
            } as Agent;
        } catch (error) {
            console.error("Error updating agent in repository:", error);
            return null;
        }
    }

    public async getAllAgents(): Promise<Agent[]> {
        try {
            const agents = await prisma.agent.findMany({
                // Only fetch agents not marked as deleted

                // where: { deleted: false }, // Alternative if null is not used
                include: { credentials: true },
                orderBy: { createdAt: "desc" },
            });
            return agents.map((agent) => ({
                ...agent,
                email: agent.email,
                agentInfo: null, // Ensure agentInfo is included
            })) as Agent[];
        } catch (error) {
            console.error("Error getting agents in repository:", error);
            throw error;
        }
    }
    public async deleteAgent(agentId: string): Promise<Agent | null> {
        try {
            const deletedAgent = await prisma.agent.delete({
                where: { id: agentId },
            });
            console.log(
                `  Agent ${agentId} and associated credentials deleted successfully.`
            );
            return {
                ...deletedAgent,
                agentInfo: null, // Ensure agentInfo is included
            } as Agent;
        } catch (error) {
            console.error("Error deleting agent in repository:", error);
            if (
                error instanceof Prisma.PrismaClientKnownRequestError &&
                error.code === "P2025"
            ) {
                console.warn(`Agent with ID ${agentId} not found for deletion.`);
                return null;
            }
            if (
                error instanceof Prisma.PrismaClientKnownRequestError &&
                error.code === "P2003"
            ) {
                console.error(
                    `  Cannot delete agent ${agentId} due to existing related records (e.g., Leads).`
                );
                throw new Error(
                    `   Cannot delete agent: Agent has related records (like Leads). Consider deactivating instead.`
                );
            }
            throw error;
        }
    }

    public async softDeleteAgent(agentId: string): Promise<Agent | null> {
        if (!agentId) {
            throw new Error("Agent ID is required for soft deletion.");
        }
        const updatedAgent = await prisma.agent.update({
            where: { id: agentId },
            data: { deleted: true },
            include: { credentials: true },
        });

        return {
            ...updatedAgent,
            agentInfo: null, // Ensure agentInfo is included
        } as Agent;
    }

    public async restoreAgent(agentId: string): Promise<Agent | null> {
        try {
            const updatedAgent = await prisma.agent.update({
                where: { id: agentId },
                data: { deleted: false }, // Set 'deleted' to false
                include: { credentials: true },
            });
            return {
                ...updatedAgent,
                agentInfo: null,
            } as Agent;
        } catch (error) {
            console.error("Error restoring agent in repository:", error);
            return null;
        }
    }
}