import { PrismaClient, QuotationStatus } from "@prisma/client";
import { Decimal } from "@prisma/client/runtime/library";

const prisma = new PrismaClient();

export class AgentQuotationRepository {
  // Utility method for handling duplicate ref numbers
  getDuplicateRefNo(originalRefNo: string, existingRefNos: string[]): string {
    const baseRefNo = originalRefNo.replace(/-D\d+$/, "");
    const duplicates = existingRefNos.filter((ref) =>
      ref.startsWith(`${baseRefNo}-D`)
    );
    const maxDuplicateNumber = duplicates.reduce((max, ref) => {
      const match = ref.match(/-D(\d+)$/);
      const num = match ? parseInt(match[1], 10) : 0;
      return Math.max(max, num);
    }, 0);
    return `${baseRefNo}-D${maxDuplicateNumber + 1}`;
  }

  async createQuotation(data: any, agentId: string) {
    try {
      const parseDate = (d: string | null | undefined) =>
        d ? new Date(d) : null;

      const generateRefNo = async () => {
        const leadId = data.leadRefNo?.replace(/^LD-/, "") ?? "UNKNOWN";
        const baseRef = `QT-${leadId}`;
        const existingQuotations = await prisma.quotation.findMany({
          where: { leadRefNo: data.leadRefNo },
          select: { refNo: true },
        });

        const revisionCount = existingQuotations.filter((q) =>
          q.refNo.startsWith(`${baseRef}-R`) || q.refNo === baseRef
        ).length;

        return revisionCount === 0 ? baseRef : `${baseRef}-R${revisionCount + 1}`;
      };
      
      const refNo = await generateRefNo();

      const newStage = "quotation_created";

      const quotation = await prisma.quotation.create({
        data: {
          refNo,
          leadRefNo: data.leadRefNo,
          leadAgent: data.leadAgent ?? null,
          status: data.status ?? QuotationStatus.Draft,
          guestName: data.guestName ?? null,
          remarks: data.remarks ?? null,
          termsAndConditions: data.termsAndConditions ?? null,
          importantNotes: data.importantNotes ?? null,
          currency: data.currency ?? null,
          currencyRate: data.currencyRate ?? null,
          totalPax: data.totalPax || null,
          travelDates: data.travelDates,
          totalBeforeMarkup: data.totalBeforeMarkup ?? null,
          totalAfterMarkup: data.totalAfterMarkup ?? null,
          agentId: agentId, // Associate with agent

          accommodations: {
            create: (data.accommodations ?? []).map((acc: any) => ({
              hotel: acc.hotel,
              hotelAddress: acc.hotelAddress,
              supplierId: acc.supplierId,
              supplierName: acc.supplierName,
              supplierContact: acc.supplierContact,
              room: acc.room,
              travelDate: parseDate(acc.travelDate),
              endDate: parseDate(acc.endDate),
              nights: acc.nights,
              surchargeNights: acc.surchargeNights,
              quantity: acc.quantity,
              totalPrice: acc.totalPrice,
              totalPriceAfterMarkup: acc.totalPriceAfterMarkup,
              priceDetailsAccommodation: {
                create: (acc.priceDetails ?? []).map((pd: any) => ({
                  category: pd.category,
                  pax: pd.pax,
                  price: pd.price,
                  priceAfterMarkup: pd.priceAfterMarkup,
                  costPrice: pd.costPrice,
                  quantity: pd.quantity,
                  totalPrice: pd.totalPrice,
                  totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
                })),
              },
            })),
          },
          itineraries: {
            create: (data.itineraries ?? []).map((itin: any) => ({
              date: parseDate(itin.date),
              label: itin.label,
              category: itin.category,
              supplierId: itin.supplierId,
              supplierName: itin.supplierName,
              supplierContact: itin.supplierContact,
              totalPriceAfterMarkup: itin.totalPriceAfterMarkup,
              priceDetailsList: {
                create: (itin.priceDetails ?? []).map((pd: any) => ({
                  category: pd.category,
                  price: pd.price,
                  priceAfterMarkup: pd.priceAfterMarkup,
                  quantity: pd.quantity,
                  totalPrice: pd.totalPrice,
                  totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
                })),
              },
            })),
          },
        },
        include: {
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
        },
      });

      // Update Lead stage and create follow-up
      const lead = await prisma.lead.findUnique({
        where: { leadNo: data.leadRefNo },
      });

      if (lead) {
        await prisma.lead.update({
          where: { id: lead.id },
          data: { stage: newStage },
        });
        await prisma.followUp.create({
          data: {
            leadId: lead.id,
            oldStage: lead.stage,
            newStage: newStage,
            notes: `Quotation ${refNo} created.`,
            userName: lead.agentName,
          },
        });
      }

      return quotation;

    } catch (error) {
      console.error("Database error while creating quotation:", error);
      throw new Error("Failed to create quotation. Please try again later.");
    }
  }

  async duplicateQuotation(id: string, userName: string) {
    try {
      const originalQuotation = await prisma.quotation.findUnique({
        where: { id },
        include: {
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          tours: true,
        },
      });

      if (!originalQuotation) {
        throw new Error("Original quotation not found for duplication");
      }

      const allQuotationsForLead = await prisma.quotation.findMany({
        where: { leadRefNo: originalQuotation.leadRefNo },
        select: { refNo: true },
      });
      const existingRefNosForLead = allQuotationsForLead.map((q) => q.refNo);
      const baseRefForDuplication = originalQuotation.refNo.replace(/-D\d+$/, "");
      const newRefNo = this.getDuplicateRefNo(baseRefForDuplication, existingRefNosForLead);

      const parseDate = (d: Date | null | undefined) => (d ? d : null);

      const duplicateData: any = {
        refNo: newRefNo,
        leadRefNo: originalQuotation.leadRefNo,
        leadAgent: originalQuotation.leadAgent,
        status: QuotationStatus.Draft,
        guestName: originalQuotation.guestName,
        remarks: `Duplicate of ${originalQuotation.refNo} (created by ${userName})`,
        termsAndConditions: originalQuotation.termsAndConditions,
        importantNotes: originalQuotation.importantNotes,
        currency: originalQuotation.currency,
        currencyRate: originalQuotation.currencyRate,
        totalBeforeMarkup: originalQuotation.totalBeforeMarkup,
        totalAfterMarkup: originalQuotation.totalAfterMarkup,
        travelDates: originalQuotation.travelDates,
        agentId: originalQuotation.agentId,
      };

      duplicateData.accommodations = {
        create: originalQuotation.accommodations.map((acc) => ({
          hotel: acc.hotel,
          hotelAddress: acc.hotelAddress,
          supplierId: acc.supplierId,
          supplierName: acc.supplierName,
          supplierContact: acc.supplierContact,
          room: acc.room,
          travelDate: parseDate(acc.travelDate),
          endDate: parseDate(acc.endDate),
          nights: acc.nights,
          surchargeNights: acc.surchargeNights,
          quantity: acc.quantity,
          totalPrice: acc.totalPrice,
          totalPriceAfterMarkup: acc.totalPriceAfterMarkup,
          priceDetailsAccommodation: {
            create: acc.priceDetailsAccommodation.map((pd) => ({
              category: pd.category,
              pax: pd.pax,
              price: pd.price,
              priceAfterMarkup: pd.priceAfterMarkup,
              costPrice: pd.costPrice,
              quantity: pd.quantity,
              totalPrice: pd.totalPrice,
              totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
            })),
          },
        })),
      };

      duplicateData.itineraries = {
        create: originalQuotation.itineraries.map((itin) => ({
          date: parseDate(itin.date),
          label: itin.label,
          category: itin.category,
          supplierId: itin.supplierId,
          supplierName: itin.supplierName,
          supplierContact: itin.supplierContact,
          totalPriceAfterMarkup: itin.totalPriceAfterMarkup,
          priceDetailsList: {
            create: itin.priceDetailsList.map((pd) => ({
              category: pd.category,
              price: pd.price,
              priceAfterMarkup: pd.priceAfterMarkup,
              quantity: pd.quantity,
              totalPrice: pd.totalPrice,
              totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
            })),
          },
        })),
      };

      duplicateData.tours = {
        connect: originalQuotation.tours.map((tour) => ({ id: tour.id })),
      };

      return await prisma.quotation.create({
        data: duplicateData,
        include: {
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          tours: true,
        },
      });
    } catch (error: any) {
      console.error("Error duplicating quotation:", error.message);
      throw new Error("Failed to duplicate quotation");
    }
  }

  async getQuotationById(id: string) {
    try {
      const quotation = await prisma.quotation.findUnique({
        where: { id },
        include: {
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          tours: true,
          lead: true,
          Agent: { include: { credentials: true } },
        },
      });
      if (!quotation) throw new Error("Quotation not found");
      return quotation;
    } catch (error) {
      console.error("Error fetching quotation:", error);
      throw error;
    }
  }

  async getAllQuotations(agentId: string) {
    try {
      return await prisma.quotation.findMany({
        where: { agentId: agentId },
        include: {
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          tours: true,
          lead: true,
          Agent: { include: { credentials: true } },
        },
        orderBy: {
            updatedAt: 'desc'
        }
      });
    } catch (error) {
      console.error("Error fetching quotations:", error);
      throw new Error("Could not fetch quotations");
    }
  }

  async updateQuotation(id: string, data: any) {
    try {
      const parseDate = (d: string | null | undefined) => (d ? new Date(d) : null);

      await prisma.priceDetailsAccommodation.deleteMany({ where: { accommodation: { quotationId: id } } });
      await prisma.accommodation.deleteMany({ where: { quotationId: id } });
      await prisma.priceDetailItinerary.deleteMany({ where: { itinerary: { quotationId: id } } });
      await prisma.itinerary.deleteMany({ where: { quotationId: id } });
      await prisma.quotation.update({ where: { id }, data: { tours: { set: [] } } });

      return await prisma.quotation.update({
        where: { id },
        data: {
          leadAgent: data.leadAgent,
          status: data.status,
          guestName: data.guestName,
          remarks: data.remarks,
          termsAndConditions: data.termsAndConditions,
          importantNotes: data.importantNotes,
          currency: data.currency,
          currencyRate: data.currencyRate,
          totalPax: data.totalPax,
          travelDates: data.travelDates,
          totalBeforeMarkup: data.totalBeforeMarkup,
          totalAfterMarkup: data.totalAfterMarkup,

          accommodations: {
            create: (data.accommodations ?? []).map((acc: any) => ({
              hotel: acc.hotel,
              hotelAddress: acc.hotelAddress,
              supplierId: acc.supplierId,
              supplierName: acc.supplierName,
              supplierContact: acc.supplierContact,
              room: acc.room,
              travelDate: parseDate(acc.travelDate),
              endDate: parseDate(acc.endDate),
              nights: acc.nights,
              surchargeNights: acc.surchargeNights,
              quantity: acc.quantity,
              totalPrice: acc.totalPrice,
              totalPriceAfterMarkup: acc.totalPriceAfterMarkup,
              priceDetailsAccommodation: {
                create: (acc.priceDetails ?? []).map((pd: any) => ({
                  category: pd.category,
                  pax: pd.pax,
                  price: pd.price,
                  priceAfterMarkup: pd.priceAfterMarkup,
                  costPrice: pd.costPrice,
                  quantity: pd.quantity,
                  totalPrice: pd.totalPrice,
                  totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
                })),
              },
            })),
          },
          itineraries: {
            create: (data.itineraries ?? []).map((itin: any) => ({
              date: parseDate(itin.date),
              label: itin.label,
              category: itin.category,
              supplierId: itin.supplierId,
              supplierName: itin.supplierName,
              supplierContact: itin.supplierContact,
              totalPriceAfterMarkup: itin.totalPriceAfterMarkup,
              priceDetailsList: {
                create: (itin.priceDetails ?? []).map((pd: any) => ({
                  category: pd.category,
                  price: pd.price,
                  priceAfterMarkup: pd.priceAfterMarkup,
                  quantity: pd.quantity,
                  totalPrice: pd.totalPrice,
                  totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
                })),
              },
            })),
          },
          tours: {
            connect: (data.tours ?? []).map((tourId: string) => ({ id: tourId })),
          },
        },
        include: {
          accommodations: { include: { priceDetailsAccommodation: true } },
          itineraries: { include: { priceDetailsList: true } },
          tours: true,
          lead: true,
        },
      });
    } catch (error) {
      console.error("Update failed:", error);
      throw new Error("Failed to update quotation. Please try again.");
    }
  }

  async deleteQuotation(id: string) {
    try {
      const existingQuotation = await prisma.quotation.findUnique({ where: { id } });
      if (!existingQuotation) throw new Error("Quotation not found");
      return await prisma.quotation.delete({ where: { id } });
    } catch (error: any) {
      console.error("Delete failed:", error);
      throw new Error("Failed to delete quotation.");
    }
  }

  async updateQuotationStatus(id: string, status: QuotationStatus, userName: string) {
    try {
      const updatedQuotation = await prisma.quotation.update({
        where: { id },
        data: { status },
        include: { lead: true },
      });
  
      if (!updatedQuotation) throw new Error("Quotation not found");
  
      if (status === 'Accepted' && updatedQuotation.lead) {
        await prisma.lead.update({
          where: { id: updatedQuotation.lead.id },
          data: {
            stage: 'quotation_accepted',
            followUps: {
              create: {
                oldStage: updatedQuotation.lead.stage,
                newStage: 'quotation_accepted',
                notes: `Quotation ${updatedQuotation.refNo} accepted by agent.`,
                userName: userName,
              },
            },
          },
        });
      }
  
      return updatedQuotation;
    } catch (error) {
      console.error("Error updating quotation status:", error);
      throw error;
    }
  }
}