import { LeadStage, LeadStatus } from "@prisma/client";
import prisma from "../../config/prismaClient";
import { Lead } from "../../interfaces/Lead";
import { NotificationService } from "../../services/common/notificationService"; // Import NotificationService


export class AgentLeadRepository {

  private notificationService: NotificationService;

  constructor() {
    this.notificationService = new NotificationService(); // Initialize NotificationService
  }

  private formatLeadDates(lead: any): Lead {
    return {
      ...lead,
      travelDateFrom: lead.travelDateFrom
        ? new Date(lead.travelDateFrom).toISOString()
        : null,
      travelDateTo: lead.travelDateTo
        ? new Date(lead.travelDateTo).toISOString()
        : null,
        createdOn: lead.createdOn // Fixes issue with missing createdOn property
            ? new Date(lead.createdOn).toISOString() : null, // Apply same logic to other date fields
    };
  }

  public async createLead(leadData: Lead): Promise<Lead> {
    try {
      const newLead = await prisma.lead.create({
        data: {
          ...leadData,
          followUps: {
            create: {
              oldStage: 'new_lead',
              newStage: 'new_lead',
              notes: 'Lead created',
              userName: leadData.agentName, // or get from context
            }
          },
        },
        include: {
          followUps: true,
        },
      });
      console.log("New lead created in agent lead repository:", newLead);

        // Notify Admin
        await this.notificationService.notifyAdmin({
            type: "lead_created",
            title: "New Agent Lead Created",
            message: `New lead created by agent ${leadData.agentName || 'Unknown Agent'}.`,
            entityType: "lead",
            entityId: newLead.id,
        });

      return this.formatLeadDates(newLead);
    } catch (error) {
      console.error("Error creating lead in agent lead repository:", error);
      throw error;
    }
  }

  //... all your other code remains the same


  public async getAgentLeads(agentId: string): Promise<Lead[]> {
    try {
      const agent = await prisma.agent.findUnique({
        where: { id: agentId },
        select: { agentCode: true },
      });

      if (!agent || !agent.agentCode) {
        throw new Error(
          "Agent not found or agentCode is missing. Agent ID:" + agentId
        );
      }

      const leads = await prisma.lead.findMany({
        where: {
          agentCode: agent.agentCode,
        },
        include: {
          followUps: true,
        },
      });

      return leads.map((lead) => this.formatLeadDates(lead));
    } catch (error) {
      console.error("Error getting agent leads in repository:", error);
      throw error;
    }
  }

  public async getLeadById(leadId: string, agentId: string): Promise<Lead | null> {
    try {
      const agent = await prisma.agent.findUnique({
        where: { id: agentId },
        select: { agentCode: true },
      });

      if (!agent || !agent.agentCode) {
        throw new Error(
          "Agent not found or agentCode is missing. Agent ID:" + agentId
        );
      }

      const lead = await prisma.lead.findUnique({
        where: {
          id: leadId,
          agentCode: agent.agentCode, // Agent can only see their leads
        },
        include: {
          followUps: true,
        },
      });
      return lead ? this.formatLeadDates(lead) : null;
    } catch (error) {
      console.error(
        "Error getting lead by ID in agent lead repository:",
        error
      );
      throw error;
    }
  }

  public async updateLead(
    leadId: string,
    agentId: string,
    leadData: Partial<Lead>
  ): Promise<Lead | null> {
    try {
      const agent = await prisma.agent.findUnique({
        where: { id: agentId },
        select: { agentCode: true },
      });

      if (!agent || !agent.agentCode) {
        throw new Error(
          "Agent not found or agentCode is missing. Agent ID:" + agentId
        );
      }
      const updatedLead = await prisma.lead.update({
        where: {
          id: leadId,
          agentCode: agent.agentCode, // Agent can only see their leads
        },
        data: leadData,
      });
      return updatedLead ? this.formatLeadDates(updatedLead) : null;
    } catch (error) {
      console.error("Error updating lead in agent lead repository:", error);
      return null;
    }
  }

  public async deleteLead(leadId: string, agentId: string): Promise<void> {
    try {
      const agent = await prisma.agent.findUnique({
        where: { id: agentId },
        select: { agentCode: true },
      });

      if (!agent || !agent.agentCode) {
        throw new Error(
          "Agent not found or agentCode is missing. Agent ID:" + agentId
        );
      }
      await prisma.lead.delete({
        where: {
          id: leadId,
          agentCode: agent.agentCode, // Agent can only see their leads
        },
      });
    } catch (error) {
      console.error("Error deleting lead in agent lead repository:", error);
      throw error;
    }
  }

  public async updateLeadStatus(id: string, status: LeadStatus): Promise<Lead | null> {
    try {
      const updatedLead = await prisma.lead.update({
        where: { id },
        data: { status },
      });
      return this.formatLeadDates(updatedLead);
    } catch (error) {
      console.error("Error updating lead status:", error);
      return null;
    }
  }
  
  public async updateLeadStage(
    id: string,
    newStage: LeadStage,
    userName?: string,
    notes?: string
  ): Promise<Lead | null> {
    try {
      // Get existing lead to fetch oldStage
      const existingLead = await prisma.lead.findUnique({
        where: { id },
        select: { stage: true },
      });
  
      if (!existingLead) return null;
  
      // Optional: Validate userId exists in DB to avoid FK violation
    
  
      const updatedLead = await prisma.lead.update({
        where: { id },
        data: {
          stage: newStage,
          followUps: {
            create: {
              oldStage: existingLead.stage,
              newStage: newStage,
              userName: userName,
              notes: notes || '',
            },
          },
        },
        include: { followUps: true },
      });
  console.log(updatedLead)
      return this.formatLeadDates(updatedLead);
    } catch (error) {
      console.error("Error updating lead stage:", error);
      return null;
    }
  }
}