import { Omit } from "utility-types";
import prisma from "../../config/prismaClient";
import { Package } from "../../interfaces/Package";



export class PackageRepository {
    // backend/src/repositories/packageRepository.ts
    public async createPackage(packageData: Omit<Package, "id">): Promise<Package> {
        try {
            const newPackage = await prisma.package.create({
                data: {
                    packageName: packageData.packageName,
                    tagLine: packageData.tagLine,
                    briefDescription: packageData.briefDescription,
                    detailedDescription: packageData.detailedDescription,
                    terms: packageData.terms,
                    notes: packageData.notes,
                    remarks: packageData.remarks,
                    imageUrl: packageData.imageUrl,
                    starRating: packageData.starRating,
                    price: packageData.price,
                    hotel: packageData.hotel,
                    daysNights: packageData.daysNights,
                    members: packageData.members,
                    deleted: packageData.deleted,
                    hotelId: packageData.hotelId,
                }
            });
            return newPackage;
        } catch (error) {
            console.error("Error creating package in repository:", error);
            throw error;
        }
    }

    public async updatePackage(
        packageId: string,
        packageData: Partial<Package>
    ): Promise<Package> {
        try {
            const updatedPackage = await prisma.package.update({
                where: { id: parseInt(packageId) },
                data: {
                    packageName: packageData.packageName,
                    tagLine: packageData.tagLine,
                    briefDescription: packageData.briefDescription,
                    detailedDescription: packageData.detailedDescription,
                    terms: packageData.terms,
                    notes: packageData.notes,
                    remarks: packageData.remarks,
                    imageUrl: packageData.imageUrl,
                    starRating: packageData.starRating,
                    price: packageData.price,
                    hotel: packageData.hotel,
                    daysNights: packageData.daysNights,
                    members: packageData.members,
                    deleted: packageData.deleted,
                    hotelId: packageData.hotelId,
                },
            });
            return updatedPackage;
        } catch (error) {
            console.error("Error updating package in repository:", error);
            throw error;
        }
    }

    public async getAllPackages(): Promise<Package[]> {
        try {
            const packages = await prisma.package.findMany();
            return packages;
        } catch (error) {
            console.error("Error getting packages in repository:", error);
            throw error;
        }
    }

    public async getPackageById(packageId: string): Promise<Package | null> {
        try {
            const packageItem = await prisma.package.findUnique({
                where: { id: parseInt(packageId) },
            });
            return packageItem;
        } catch (error) {
            console.error("Error getting package by ID in repository:", error);
            throw error;
        }
    }

    public async deletePackage(packageId: string): Promise<void> {
        try {
            await prisma.package.delete({
                where: { id: parseInt(packageId) },
            });
        } catch (error) {
            console.error("Error deleting package in repository:", error);
            throw error;
        }
    }
}
