import { PrismaClient } from "@prisma/client";


import cuid from 'cuid';
import { Hotel } from "../../interfaces/Hotel";

const prisma = new PrismaClient({
  log: ['query', 'info', 'warn', 'error'],
});

export class HotelRepository {
  
  async createHotel(data: Hotel) {
    try {
      console.log("Data being passed to createHotel:", JSON.stringify(data, null, 2));
  
      const createdHotel = await prisma.hotels.create({
        data: {
          id: data.id ?? cuid(), // Use provided ID or generate a new one
          name: data.name ?? "",
          location: data.location ?? "",
          starRating: data.starRating ?? "",
          tourismTax: data.tourismTax ?? "",
          stateTax: data.stateTax ?? "",
          address: data.address ?? "",
          phone: data.phone ?? "",
          description: data.description ?? "",
          termsAndConditions: data.termsAndConditions ?? "",
          supplierId: data.supplierId ?? null,
          surcharge: data.surcharge ?? {},
          closedDates: data.closedDates ?? {},
          price: data.price
            ? {
                create: data.price.map((p) => ({
                  category: p.category ?? "",
                  pax: p.pax ?? "",
                  costPrice: p.costPrice ?? 0,
                  optionalPrice: p.optionalPrice ?? 0,
                })),
              }
            : undefined,
        },
      });
  
      console.log("Hotel created successfully:", createdHotel);
      return createdHotel;
    } catch (error) {
      console.error("Error creating hotel:", error);
      throw new Error("Failed to create hotel");
    }
  }
  
  async getHotels(page = 1, limit = 10, searchTerm = "") {
    try {
      const skip = (page - 1) * limit;
  
      const hotels = await prisma.hotels.findMany({
        where: {
          name: {
            contains: searchTerm,
            mode: "insensitive",
          },
        },
        orderBy: {
          createdAt: "desc",
        },
        skip,
        take: limit,
        include: {
          rooms: true,
          price: true,
          supplier: {
            select: {
              id: true,
              companyName: true, // Fetching only required supplier details
            },
          },
        },
      });
  
      const totalHotels = await prisma.hotels.count({
        where: {
          name: {
            contains: searchTerm,
            mode: "insensitive",
          },
        },
      });
  
      return {
        hotels,
        currentPage: page,
        totalPages: Math.ceil(totalHotels / limit),
        totalItems: totalHotels,
      };
    } catch (error) {
      console.error("Error fetching hotels:", error);
      throw new Error("Failed to fetch hotels");
    }
  }
  

  async getHotelById(id: string) {
    try {
      return await prisma.hotels.findUnique({
        where: { id },
        include: {
          rooms: true, // Include associated rooms
          price:true,
          supplier: true,
        },
      });
    } catch (error) {
      console.error(`Error fetching hotel with ID ${id}:`, error);
      throw new Error("Failed to fetch hotel");
    }
  }
  
  async updateHotel(id: string, data: Hotel) {
    try {
      return await prisma.hotels.update({
        where: { id },
        data: {
          name: data.name ?? "",
          location: data.location ?? "",
          starRating: data.starRating ?? "",
          tourismTax: data.tourismTax ?? "",
          stateTax: data.stateTax ?? "",
          address: data.address ?? "",
          phone: data.phone ?? "",
          description: data.description ?? "",
          termsAndConditions: data.termsAndConditions ?? "",
          supplierId: data.supplierId ?? "",
  
          // Correctly updating JSON fields
          surcharge: data.surcharge ? data.surcharge : undefined,
          closedDates: data.closedDates ? data.closedDates : undefined,
  
          // Handling `price` updates
          price: {
            upsert: data.price
              ? data.price.map((p) => ({
                  where: { id: p.id ?? "" },
                  update: {
                    category: p.category ?? "",
                    pax: p.pax ?? "",
                    costPrice: p.costPrice ?? 0,
                    optionalPrice: p.optionalPrice ?? 0,
                  },
                  create: {
                    category: p.category ?? "",
                    pax: p.pax ?? "",
                    costPrice: p.costPrice ?? 0,
                    optionalPrice: p.optionalPrice ?? 0,
                   
                  },
                }))
              : undefined,
          },
        },
      });
    } catch (error: any) {
      console.error(`Error updating hotel with ID ${id}:`, error);
      throw new Error("Failed to update hotel: " + error.message);
    }
  }
  
  
  

  async deleteHotel(id: string) {
    try {
      return await prisma.hotels.delete({ where: { id } });
    } catch (error) {
      console.error(`Error deleting hotel with ID ${id}:`, error);
      throw new Error("Failed to delete hotel");
    }
  }


 async createRoom(data: any) {
    const room= await prisma.room.create({
      data: {
        name: data.name,
        hotelId: data.hotelId,
        maxPax: data.maxPax,
        extraBeds: data.extraBeds,
        prices: {
          create: data.prices.map((price: any) => ({
            category: price.category,
            pax: price.pax,
            costPrice: price.costPrice,
            optionalPrice: price.optionalPrice,
          })),
        },
      },
      include: {
        prices: true,
        hotel: true, 
      },
    });
    console.log(room)
    return room;
  }

  async updateRoom(id: string, data: any) {
    return await prisma.room.update({
      where: { id },
      data: {
        name: data.name,
        maxPax: data.maxPax,
        extraBeds: data.extraBeds,
        prices: {
          create: data.prices.filter((price: any) => !price.id).map((price: any) => ({  // Create new prices
            category: price.category,
            pax: price.pax,
            costPrice: price.costPrice,
            optionalPrice: price.optionalPrice,
          })),
          update: data.prices.filter((price: any) => price.id).map((price: any) => ({  // Update existing prices
            where: { id: price.id },
            data: {
              category: price.category,
              pax: price.pax,
              costPrice: price.costPrice,
              optionalPrice: price.optionalPrice,
            },
          })),
        },
      },
      include: {
        prices: true,
        hotel: true,
      },
    });
  }

  async deleteRoom(id: string) {
    return await prisma.room.delete({ where: { id } });
  }

  async getRoomById(id: string) {
    return await prisma.room.findUnique({
      where: { id },
      include: {
        prices: true,
        hotel:true, 
      },
    });
  }
  
  async getRoomsByHotelId(hotelId: string) {
    return await prisma.room.findMany({
      where: { hotelId },
      include: {
        prices: true, 
        hotel:true,
      },
    });
  }
}
