import { Lead,LeadStage } from "@prisma/client";
import prisma from "../../config/prismaClient";
import { subDays, startOfDay } from 'date-fns';

function getLastNDates(n: number) {
  const dates = [];
  for (let i = n - 1; i >= 0; i--) {
    dates.push(startOfDay(subDays(new Date(), i)));
  }
  return dates;
}

export class DashboardRepository {
  async getCountsPerDay(tableName: 'user' | 'agent' | 'lead', n = 7) {
    const dates = getLastNDates(n);

    const dateField = tableName === 'lead' ? 'createdOn' : 'createdAt';
    const results = await (prisma[tableName] as any).findMany({
      where: {
        [dateField]: {
          gte: dates[0],
        },
      },
      select: {
        [dateField]: true,
      },
    });

    const groupedResults = results.reduce(
      (acc: Record<string, number>, curr: { createdAt?: Date, createdOn?: Date }) => {
        const dateKey = (curr.createdAt || curr.createdOn)?.toDateString();
        if (dateKey) {
          acc[dateKey] = (acc[dateKey] || 0) + 1;
        }
        return acc;
      },
      {} as Record<string, number>
    );

    const counts = dates.map(date => {
      const found = results.find((r: { createdAt?: Date, createdOn?: Date }) =>
        (r.createdAt || r.createdOn)?.toDateString() === date.toDateString()
      );
      return groupedResults[date.toDateString()] || 0;
    });

    return counts;
  }

 async getDashboardData() {
  try {
    const totalLeads = await prisma.lead.count();
    const bookedLeads = await prisma.lead.count({
      where: {
        stage: { in: ['quotation_accepted', 'assigned_to_operations', 'assigned_to_assessment','trip_started', 'trip_completed'] },
      },
    });
    const newLeads = await prisma.lead.findMany({ where: { stage: "new_lead" } }); // ⬅️ Full data
    const userCount = await prisma.user.count();
    const agentCount = await prisma.agent.count();

    const leadsData = await this.getCountsPerDay('lead');
    const usersData = await this.getCountsPerDay('user');
    const agentsData = await this.getCountsPerDay('agent');

    const bookingsData = await prisma.lead.groupBy({
      by: ['createdOn'],
      _count: { _all: true },
      where: {
        stage: { in: ['quotation_accepted', 'assigned_to_operations', 'assigned_to_assessment'] },
        createdOn: {
          gte: getLastNDates(7)[0],
        },
      },
    });

    const dates = getLastNDates(7);
    const bookings = dates.map(date => {
      const found = bookingsData.find(r =>
        r.createdOn.toDateString() === date.toDateString()
      );
      return found ? found._count._all : 0;
    });

    return {
      totalLeads,
      bookedLeads,
      newLeads, // ⬅️ Full new lead objects returned here
      userCount,
      agentCount,
      dataSeries: {
        users: usersData,
        agents: agentsData,
        leads: leadsData,
        bookings: bookings,
      },
    };
  } catch (error: any) {
    console.error("Repository Error:", error);
    throw new Error("Failed to retrieve dashboard data");
  }
}
 async getLeadStageCounts(period: 'all' | '30d' | '90d' = 'all') {
    try {
      let dateFilter: { gte?: Date } = {};
      if (period !== 'all') {
        const days = period === '30d' ? 30 : 90;
        dateFilter = { gte: subDays(new Date(), days) };
      }

      // Define the stages you want to count
      const stagesToCount: LeadStage[] = [
        LeadStage.new_lead,
        LeadStage.quotation_created,
        LeadStage.quotation_sent,
        LeadStage.quotation_accepted,
        LeadStage.trip_started,
        LeadStage.trip_completed,
      ];

      // Use a single groupBy query for efficiency
      const countsByStage = await prisma.lead.groupBy({
        by: ['stage'],
        where: {
          stage: { in: stagesToCount }, // Only consider the stages we care about
          createdOn: dateFilter,
        },
        _count: {
          _all: true,
        },
      });
      // The result is: [{ stage: 'new_lead', _count: { _all: 150 } }, { stage: 'quotation_sent', _count: { _all: 120 } }]

      // Initialize a result object with 0 for all target stages
      const result: { [key in LeadStage]?: number } = {};
      stagesToCount.forEach(stage => {
        result[stage] = 0;
      });

      // Populate the result object with actual counts from the query
      for (const group of countsByStage) {
        result[group.stage] = group._count._all;
      }
      
      return result;

    } catch (error) {
      console.error("Repository Error - getLeadStageCounts:", error);
      throw new Error("Failed to retrieve lead stage counts from database");
    }
  }

async getTopAgentsByLeadCount(limit: number = 5) {
  try {
    return await prisma.lead.groupBy({
      by: ["agentCode", "agentName"],
      where: {
        agentCode: { not: null },
      },
      _count: {
        agentCode: true,
      },
      orderBy: {
        _count: {
          agentCode: "desc",
        },
      },
      take: limit,
    });
  } catch (error) {
    console.error("Repository Error - getTopAgentsByLeadCount:", error);
    throw new Error("Failed to retrieve top agents from database");
  }
}
// src/repositories/sales.repository.ts

  async getTopSalesByLeadCount(limit: number = 5) {
    try {
      // Step 1: Group leads by the salesperson's ID (`userId`) and count them.
      const topSalespersonsData = await prisma.lead.groupBy({
        by: ["userId"], // <-- The only change needed is here: using 'userId'
        where: {
          // We only want leads that have a userId
          userId: { not: null }, // <-- And here: filtering by 'userId'
        },
        _count: {
          _all: true, // Count all leads for each userId group
        },
        orderBy: {
          _count: {
            userId: "desc", // Order by the count of userId
          },
        },
        take: limit,
      });

      // The result from Step 1 is: [{ userId: 'user_id_1', _count: { _all: 50 } }, ...]

      if (topSalespersonsData.length === 0) {
        return []; // Return early if no relevant leads are found
      }

      // Step 2: Extract the user IDs from the result.
      const userIds = topSalespersonsData.map((item) => item.userId!); // <-- Using item.userId

      // Step 3: Fetch the user details (like name and empCode) for those user IDs.
      // This part of the logic remains the same.
      const users = await prisma.user.findMany({
        where: {
          id: { in: userIds },
        },
        select: {
          id: true,
          name: true,
          empCode: true,
        },
      });

      // Create a map for efficient lookups
      const userMap = new Map(users.map(user => [user.id, { name: user.name, empCode: user.empCode }]));

      // Step 4: Combine the user details with their lead counts.
      const topSalespersons = topSalespersonsData.map((data) => {
        const user = userMap.get(data.userId!); // <-- Using data.userId for the lookup
        return {
          userId: data.userId!,
          salesPersonName: user?.name || "Unknown User",
          empCode: user?.empCode || "N/A",
          leadCount: (typeof data._count === "object" && data._count !== null && "_all" in data._count) ? (data._count as { _all: number })._all : 0,
        };
      });
console.log("Top Salespersons Data:", topSalespersons); // Debugging log
      return topSalespersons;
    } catch (error) {
      console.error("Repository Error - getTopSalespersonsByLeadCount:", error);
      throw new Error("Failed to retrieve top salespersons from database");
    }
  }
}


