/* eslint-disable @typescript-eslint/no-unused-vars */
import { Response, NextFunction, Request } from "express";
import jwt, { JwtPayload } from "jsonwebtoken";
import { generateToken } from "../utils/generateJwt";
import "../types/express"; // Ensure Express types are extended
import dotenv from "dotenv";

dotenv.config();

const SECRET_KEY = process.env.JWT_SECRET!;
const REFRESH_SECRET = process.env.JWT_REFRESH_SECRET!;

export const authenticateUser = async (req: Request, res: Response, next: NextFunction): Promise<void> => {
  try {
    const authorizationHeader = req.headers.authorization;
    let refreshTokenHeader = req.headers["x-refresh-token"];

    if (!authorizationHeader) {
      res.status(403).json({ success: false, message: "Access token is missing" });
      return;
    }

    const token = authorizationHeader.split(" ")[1];

    try {
      const decoded = jwt.verify(token, SECRET_KEY) as JwtPayload;

      if (decoded && decoded.userId && decoded.role) {
        req.currentUser = decoded.userId;
        req.currentUserRole = decoded.role;
        req.decoded = decoded; // Store the decoded token

        return next();
      }
      res.status(403).json({ success: false, message: "Invalid token" });
      return;
    } catch (error) {
      if ((error as Error).name === "TokenExpiredError" && refreshTokenHeader) {
        // ✅ Ensure refreshTokenHeader is a string
        if (Array.isArray(refreshTokenHeader)) {
          refreshTokenHeader = refreshTokenHeader[0]; // Take the first value
        }

        if (typeof refreshTokenHeader !== "string") {
          res.status(403).json({ success: false, message: "Invalid refresh token format" });
          return;
        }

        try {
          const refreshDecoded = jwt.verify(refreshTokenHeader, REFRESH_SECRET) as JwtPayload;
          if (refreshDecoded && refreshDecoded.userId && refreshDecoded.role) {
            req.currentUser = refreshDecoded.userId;
            req.currentUserRole = refreshDecoded.role;
            req.decoded = refreshDecoded;

            const newAccessToken = generateToken(refreshDecoded.userId, refreshDecoded.role, SECRET_KEY, "1h");
            res.setHeader("x-access-token", newAccessToken);
            return next();
          }
          res.status(403).json({ success: false, message: "Invalid refresh token" });
          return;
        } catch (refreshError) {
          res.status(403).json({ success: false, message: "Invalid or expired refresh token" });
          return;
        }
      } else {
        res.status(403).json({ success: false, message: "Invalid access token" });
        return;
      }
    }
  } catch (error: unknown) {
    console.error("Authentication Error:", error);
    res.status(403).json({ success: false, message: "Unauthorized access" });
    return;
  }
};