import { Request, Response } from "express";
import { UserService } from "../../services/user/userService";

const userService = new UserService();

export class UserController {
  // In UserController
  async createUser(req: Request, res: Response) {
    try {
      const user = await userService.createUser(req.body);
      console.log(user, "user in controller");
      res.status(201).json(user);
    } catch (error: unknown) {
      if (error instanceof Error) {
        if (error.message.includes("already exists")) {
          res.status(409).json({ message: error.message }); // 409 Conflict
        } else {
          res.status(400).json({ message: error.message }); // 400 Bad Request
        }
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  async updateUser(req: Request, res: Response) {
    try {
      const user = await userService.updateUser(req.params.id, req.body);
      res.json(user);
    } catch (error: unknown) {
      if (error instanceof Error) {
        if (error.message.includes("already exists")) {
          res.status(409).json({ message: error.message }); // 409 Conflict
        } else {
          res.status(400).json({ message: error.message }); // 400 Bad Request
        }
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }
  async getUsers(req: Request, res: Response) {
    try {
      const users = await userService.getUsers();
      res.json(users);
    } catch (error: unknown) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  async getUserById(req: Request, res: Response) {
    try {
      const user = await userService.getUserById(req.params.id);
      if (!user) {
        res.status(404).json({ message: "User not found" });
        return;
      }
      res.json(user);
    } catch (error: unknown) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  async getUserByEmail(req: Request, res: Response) {
    try {
      const user = await userService.getUserByEmail(req.params.email);
      if (!user) {
        res.status(404).json({ message: "User not found" });
        return;
      }
      res.json(user);
    } catch (error: unknown) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  async deleteUser(req: Request, res: Response) {
    try {
      await userService.deleteUser(req.params.id);
      res.json({ message: "User deleted successfully" });
    } catch (error: unknown) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }
  async restore(req: Request, res: Response) {
    try {
      const userId = req.params.id;
      const restoredUser = await userService.restore(userId);

      if (!restoredUser) {
        return res.status(404).json({ message: "User not found" });
      }

      return res.status(200).json({ message: "User restored successfully" });
    }
    catch (error: unknown) {
      if (error instanceof Error) {
        return res.status(500).json({ message: error.message });
      } else {
        return res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  async deleteUserPermanently(req: Request, res: Response) {
    try {
      const userId = req.params.id;
      const deletedUser = await userService.deleteUserPermanently(userId);

      if (!deletedUser) {
        return res
          .status(404)
          .json({ message: "User not found or already deleted" });
      }

      return res.status(200).json({ message: "User permanently deleted" });
    } catch (error: unknown) {
      if (error instanceof Error) {
        return res.status(500).json({ message: error.message });
      } else {
        return res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  async getInactiveUsers(req: Request, res: Response) {
    try {
      const users = await userService.getInactiveUsers();
      res.json(users);
    } catch (error: unknown) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  async getUsersByTeamId(req: Request, res: Response) {
    try {
      const teamId= req.params.id
      console.log(teamId)
      const users = await userService.getUsersByTeamId(teamId);
      res.json(users);
    } catch (error: unknown) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  async searchUsers(req: Request, res: Response) {
    try {
      const users = await userService.searchUsers(
        req.query.searchTerm as string
      );
      res.json(users);
    } catch (error: unknown) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }
}
