import { Request, Response } from 'express';
import { QuotationService } from '../../services/admin/quotationService';
import { BookingService } from '../../services/admin/bookingService'; 
import { QuotationRepository } from '../../repositories/admin/quotationRepository';


export class QuotationController {
  private quotationService: QuotationService;
  private bookingService: BookingService; // Add this

  constructor(
    quotationService?: QuotationService,
    bookingService?: BookingService
  ) {
    this.quotationService = quotationService || new QuotationService(new QuotationRepository());
    this.bookingService = bookingService || new BookingService(); 
  }
  async createQuotation(req: Request, res: Response) {
    try {
      const {data,userName} = req.body;
      console.log("Request Body:", data);
  
      const quotation = await this.quotationService.createQuotation(data,userName);
      console.log("Quotation created successfully:", quotation);
      res.status(201).json(quotation);
    } catch (error) {
      console.error("Controller error while creating quotation:", error);
      const errorMessage =
        error instanceof Error
          ? error.message
          : "Internal server error while creating quotation.";
      res.status(500).json({ error: errorMessage });
    }
  }
  

  async getQuotationById(req: Request, res: Response) {
    try {
      const { id } = req.params;
      console.log("Fetching quotation for ID:", id);

      if (!id) {
        return res.status(200).json([]);
      }

      const quotation = await this.quotationService.getQuotationById(id);
      console.log(quotation, "quotation in controller getQuotationbyid");
      if (!quotation) {
        return res.status(200).json([]);
      }

      res.status(200).json(quotation);
    } catch (error) {
      console.error("Controller error:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }

  async getQuotationByLeadRefNo(req: Request, res: Response) {
    try {
      const { leadRefNo } = req.params;

      if (!leadRefNo) {
        return res.status(200).json({ message: "Lead reference number is required" });
      }
      // Log the leadRefNo for debugging
      if (leadRefNo === "undefined") {
        console.error("Received 'undefined' for leadRefNo");
        return res.status(400).json({ message: "Invalid lead reference number" });
      }

      // Fetch the quotation using the service




      console.log("Fetching quotation for leadRefNo:", leadRefNo);
      const quotation = await this.quotationService.getQuotationByLeadRefNo(leadRefNo);
      res.status(200).json(quotation);
    } catch (error) {
      console.error("Error in controller:", error);
      res.status(500).json({ message: "Failed to fetch quotation", error });
    }
  }             

async getQuotations(req: Request, res: Response) {
  try {
    const quotations = await this.quotationService.getAllQuotations();
    return res.status(200).json(quotations || []);
  } catch (error) {
    console.error("Error fetching quotations:", error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
}


  async updateQuotation(req: Request, res: Response) {
    try {
      const { id } = req.params;
      const data = req.body;

      console.log("Received update request for quotation ID:", id);
      console.log("Update data:", JSON.stringify(data, null, 2));

      // Validate if ID is provided
      if (!id) {
        return res.status(400).json({ error: "Quotation ID is required" });
      }

      // Validate if request body contains data
      if (!data || Object.keys(data).length === 0) {
        return res.status(400).json({ error: "Update data is required" });
      }

      // Call the service method to update the quotation
      const updatedQuotation = await this.quotationService.updateQuotation(
        id,
        data
      );

      console.log("Updated Quotation", updatedQuotation);

      // If no quotation is found with the given ID
      if (!updatedQuotation) {
        return res
          .status(404)
          .json({ error: `No quotation found with ID: ${id}` });
      }

      console.log(
        "Successfully updated quotation:",
        JSON.stringify(updatedQuotation, null, 2)
      );

      res.status(200).json(updatedQuotation);
    } catch (error: any) {
      console.error("Error updating quotation:", error);

      // Handle specific errors
      if (error.name === "ValidationError") {
        return res
          .status(400)
          .json({ error: "Invalid data format", details: error.message });
      }
      if (error.name === "MongoError") {
        return res
          .status(500)
          .json({ error: "Database error", details: error.message });
      }
      if (error.code === "ECONNREFUSED") {
        return res.status(503).json({ error: "Database connection refused" });
      }

      res
        .status(500)
        .json({ error: "Internal server error", details: error.message });
    }
  }

  async deleteQuotation(req: Request, res: Response) {
    try {
      const { id } = req.params;
      console.log("Received ID:", id); // Debugging

      if (!id) {
        return res.status(400).json({ error: "Quotation ID is required" });
      }

      const deletedQuotation = await this.quotationService.deleteQuotation(id);
      if (!deletedQuotation) {
        return res.status(404).json({ error: "Quotation not found" });
      }

      res.status(200).json({ message: "Quotation deleted successfully" });
    } catch (error) {
      console.error("Error deleting quotation:", error); // Debugging
      res.status(500).json({ error: "Error deleting quotation" });
    }
  }

  async updateQuotationStatus(req: Request, res: Response) {
    try {
      const { id } = req.params;
      const { status, userName } = req.body;

      if (!id || !status) {
        return res.status(400).json({ message: "Quotation ID and status are required" });
      }

      // Step 1: Update the quotation status
      const updatedQuotation = await this.quotationService.updateQuotationStatus(id, status, userName);
      console.log("Updated quotation status in controller:", updatedQuotation);

      let booking = null;

      // Step 2: If status is Accepted, create a booking
      if (status === "Accepted") {
        booking = await this.bookingService.createBookingFromQuotation(updatedQuotation.id);
        console.log("Booking created from accepted quotation:", booking);
      }

      return res.status(200).json({
        message: "Quotation status updated successfully",
        quotation: updatedQuotation,
        booking: booking || null,
      });
    } catch (error: any) {
      console.error("Error in controller while updating quotation status:", error);
      return res.status(500).json({ message: error.message || "Failed to update quotation status" });
    }
  }
  async getAcceptedQuotationsByLeadId (req: Request, res: Response)  {
  try {
    const { leadId } = req.params;
    const data = await this.quotationService.getAcceptedQuotationsByLeadId(leadId);
    res.status(200).json(data);
  } catch (error:any) {
    res.status(500).json({ message: error.message || "Failed to fetch data" });
  }
}

async duplicateQuotation(req: Request, res: Response) {
  try {
    const { id } = req.params;
    const { userName } = req.body;

    if (!id) {
      return res.status(400).json({ message: "Quotation ID is required" });
    }

    const duplicatedQuotation = await this.quotationService.duplicateQuotation(id, userName);
    console.log("Duplicated quotation in controller:", duplicatedQuotation);

    return res.status(200).json(duplicatedQuotation);
  } catch (error:any) {
    console.error("Error in controller while duplicating quotation:", error);
    return res.status(500).json({ message: error.message || "Failed to duplicate quotation" });
  }
}

}
